<?php namespace Concore\Articles\Services;

use Concore\Articles\Models\Article;
use Concore\Articles\Models\Story;

class ArticlesService {

	protected $article;
	protected $story;

	public function __construct(Article $article, Story $story) {
		$this->article = $article;
		$this->story = $story;
	}

	public function getLatest() {
		return $this->article->orderByDate('DESC')->paginate(10);
	}

	public function getByKey($key) {
		return $this->article->find($key);
	}

	public function getRelated(Article $article, $n = null) {
		// First attempt to get $n actually related articles
		// Passing 0 and null to slice() just causes it to get all from collection
		$related = $article->related->slice(0, $n);
		// If we've specified $n and we have less than $n, pad out with random articles
		if($n && $related->count() < $n) {
			$random = $this->article->whereNotIn('id',array_merge([$article->id],array_pluck($related,'id')))
							 ->orderByRaw('RAND()')
							 ->take($n - $related->count())
							 ->get();
			$random->each(function($item) use($related) {
				$related->push($item);
			});
		}
		return $related;
	}

	public function getFeaturedStory() {
		$stories = $this->story->recentlyUpdated()->get();
		return $stories->first();
	}

	public function getArchive($year_month) {
		list($year,$month) = explode('-',$year_month);
		$from = \Carbon\Carbon::createFromDate($year,$month, 1)->setTime(0,0,0);
		$to = \Carbon\Carbon::createFromDate($year,$month + 1, 1)->setTime(0,0,0);
		return $this->article->whereBetween('date',[$from,$to])->orderByDate()->paginate(10);
	}

	public function getArchiveList() {
		$articles = $this->article->orderByDate()->select('date')->get();
		$archives = [];
		foreach($articles as $article) {
			if(!isset($archives[$article->date->format('F Y')])) {
				$archives[$article->date->format('F Y')] = route('articles.archive',$article->date->year . '-' . $article->date->format('m'));
			}
		}
		return $archives;
	}

	public function findWithTag($tag) {
		$tags_service = app('Concore\Tags\Services\TagsService');
		return $tags_service->findWithTag($this->article, $tag);
	}

}
