<?php namespace Concore\Auditing\Services;

use Auth, Config, DB;
use Carbon\Carbon;
use Concore\Auditing\AuditLogCollection;
use Concore\Auditing\CompositeAuditLog;
use Concore\Auditing\Models\AuditLog;

class AuditLogService {

    public function __construct(AuditLog $audit_log) {
        $this->audit_log = $audit_log;
    }

    public function put($type, $loggee = null, $data = null) {
        $this->audit_log->create([
            'logger_user_id' => Auth::id(),
            'type' => $type,
            'loggee_type' => $loggee ? get_class($loggee) : null,
            'loggee_id' => $loggee ? $loggee->id : null,
            'data' => $data ? serialize($data) : null,
            'created_at' => Carbon::now()
        ]);
    }

    public function putComposite($composite) {
        DB::transaction(function() use($composite) {
            foreach($composite as $log) {
                $type = $log[0];
                $loggee = isset($log[1]) ? $log[1] : null;
                $data = isset($log[2]) ? $log[2] : null;
                $this->put($type,$loggee,$data);
            }
        });
    }

    public function has() {
        return $this->audit_log->count() > 0;
    }

    public function all($n = null) {
        return $this->getAuditLogs($n);
    }

    public function forLogger($logger, $type = null, $n = null) {
        return $this->getAuditLogs($n, $type, function($query) use($logger) {
            return $query->forLogger($logger)->dateOrder();
        });
    }

    public function forLoggee($loggee, $type = null, $n = null) {
        return $this->getAuditLogs($n, $type, function($query) use($loggee) {
            return $query->forLoggee($loggee);
        });
    }

    protected function getAuditLogs($n = null, $type = null, $closure = null) {
        $query = $this->audit_log;
        if($closure) {
            $query = $closure($query);
        }
        if($n) {
            $query->take($n);
        }
        if($type) {
            $query->whereType($type);
        }
        $audit_logs = $query->orderBy('id','desc')->get();
        // Parse audit_logs for composite logs
        if(Config::get('auditing.enable_composites') && Config::has('auditing.composites') && !empty(Config::get('auditing.composites'))) {
            $this->parseComposites($audit_logs);
        }
        return new AuditLogCollection($audit_logs);
    }

    protected function parseComposites(&$audit_logs) {
        $composites = Config::get('auditing.composites');
        for($i = 0; $i < count($audit_logs); $i++) {
            foreach($composites as $composite_type => $sub_types) {
                // Because we're looping through audit logs in reverse chronological order, it means we'll come across
                // the last sub-type in a composite before the first one, since they're entered in chronological order
                if($audit_logs[$i]->type === end($sub_types)) {
                    // This is the end of the composite audit log $composite, which means
                    // the next count($sub_types) should belong to that audit log
                    $next_relevant_logs = $audit_logs->slice($i, count($sub_types));
                    $this_types = array_reverse(array_pluck($next_relevant_logs->toArray(),'type'));
                    if($this_types !== $sub_types) {
                        // Something has gone wrong because logs aren't in database in correct order to match up with composites, database is corrupt
                        // Just ignore these for now and leave un-composited logs in the array
                    } else {
                        $composite = new CompositeAuditLog($next_relevant_logs);
                        $composite->type = $composite_type;
                        $composite->sortBy(function($item) {
                            return $item->id;
                        });
                        $audit_logs->splice($i,count($sub_types),[$composite]);
                    }
                }
            }
        }
    }

}