<?php namespace Concore\Auditing\Models;

use View, Config, Lang, Log;
use Illuminate\Database\Eloquent\Model;

class AuditLog extends Model {

    public $timestamps = false;

	protected $guarded = ['id'];
    protected $dates = ['created_at'];
    protected $casts = ['loggee'=>'array'];

    public function __toString() {
        try {
            $view = 'auditing::audit-log.' . $this->type;
            if(!View::exists($view)) {
                $view = 'auditing::audit-log.default';
            }
            $string = view($view)->with('log',$this)->__toString();
        }
        catch (\Exception $e) {
            $message = "exception '".get_class($e)."' with message ".$e->getMessage()." in ".$e->getFile().":".$e->getLine();
            Log::error($message);
            if (env('APP_DEBUG')) {
                return $message;
            }
            return '';
        }
        return $string;
    }

    public function logger() {
        return $this->belongsTo(app('Concore\Foundation\Models\User'),'logger_user_id');
    }

    public function loggee() {
        return $this->morphTo();
    }

    public function getDataAttribute() {
        return unserialize($this->attributes['data']);
    }

    public function getTypeHumanReadableAttribute() {
        if (Lang::has('auditing::messages.' . $this->type)) {
            return Lang::get('auditing::messages.' . $this->type);
        }
        return str_replace('-', ' ', $this->type);
    }

    public function getLoggeeTypeHumanReadableAttribute() {
        if(method_exists($this->loggee,'getHumanReadable')) {
            return $this->loggee->getHumanReadable();
        } else {
            $debris = explode("\\",$this->loggee_type);
            return strtolower(array_pop($debris));
        }
    }

    public function scopeDateOrder($query) {
        return $query->orderBy('created_at','desc');
    }

    public function scopeReverseDateOrder($query) {
        return $query->orderBy('created_at','asc');
    }

    public function scopeForLogger($query, $logger) {
        if (is_object($logger)) {
            $logger_id = $logger->id;
        }
        else {
            $logger_id = $logger;
        }
        return $query->where('logger_user_id','=',$logger_id);
    }

    public function scopeForLoggee($query, $model) {
        if (is_array($model)) {
            //add each loggee that you want to include, add each to query
            foreach($model as $one_model) {
                $query = $query->orWhere(function($query) use($one_model) {
                    return $query->forLoggee($one_model);
                });
            }
            return $query;
        }
        else {
            return $query->where('loggee_type','=',get_class($model))->where('loggee_id','=',$model->id);
        }
    }

    public function scopeForLoggerAndLoggee($query, $logger, $loggee, $inclusive) {
        if ($inclusive) {
            return $query->where(function($query) use($logger, $loggee) {
                $query = $query->orWhere(function($query) use($logger) {
                    return $query->forLogger($logger);
                });
                $query = $query->orWhere(function($query) use($loggee) {
                    return $query->forLoggee($loggee);
                });
                return $query;
            });
        }
        else {
            return $query->where(function($query) use($logger, $loggee) {
                $query = $query->where(function($query) use($logger) {
                    return $query->forLogger($logger);
                });
                $query = $query->where(function($query) use($loggee) {
                    return $query->forLoggee($loggee);
                });
                return $query;
            });
        }
    }

    public function scopeForComposite($query, $composite_type) {
        return $query->whereIn('type',Config::get('auditing.composites.' . $composite_type));
    }

    public function newCollection(array $models = Array()) {
        return app('Concore\Auditing\Models\Collections\AuditCollection', [$models]);
    }

    public function getLoggerNameAttribute($logger_name) {
        if ($this->logger) {
            $logger_name = $this->logger->name;
        }
        return $logger_name;
    }

}