<?php namespace Concore\Auditing\Services;

use Auth, Config, DB;
use Carbon\Carbon;
use Concore\Auditing\AuditLogCollection;
use Concore\Auditing\AuditLogLengthAwarePaginator;
use Concore\Auditing\CompositeAuditLog;
use Concore\Auditing\Models\AuditLog;

class AuditLogService {

    private $logger = null;
    private $loggee = null;
    private $types = null;

    public function __construct(AuditLog $audit_log) {
        $this->audit_log = $audit_log;
    }

    private function setLoggerLoggeeAndTypes($logger = null, $loggee = null, $types = null) {
        $this->logger = $logger;
        $this->loggee = $loggee;
        $this->types = $types;
    }

    public function put($type, $loggee = null, $data = null) {
        $this->audit_log->create([
            'logger_user_id' => Auth::id(),
            'logger_name' => Auth::user()->name,
            'type' => $type,
            'loggee_type' => $loggee ? get_class($loggee) : null,
            'loggee_name' => $loggee ? $this->getLoggeeNameAttribute($loggee)  : null,
            'loggee_id' => $loggee ? $loggee->id : null,
            'data' => $data ? serialize($data) : null,
            'created_at' => Carbon::now()
        ]);
    }

    public function putComposite($composite) {
        DB::transaction(function() use($composite) {
            foreach($composite as $log) {
                $type = $log[0];
                $loggee = isset($log[1]) ? $log[1] : null;
                $data = isset($log[2]) ? $log[2] : null;
                $this->put($type,$loggee,$data);
            }
        });
    }

    public function has() {
        return $this->audit_log->count() > 0;
    }

    public function all($n = null) {
        return $this->getAuditLogs($n);
    }

    public function paginate($per_page = null, $type = null) {
        $this->setLoggerLoggeeAndTypes(null, null, $type);
        return $this->getAuditLogsPaginated($per_page, $type);
    }

    public function forLogger($logger, $type = null, $n = null) {
        $this->setLoggerLoggeeAndTypes($logger, null, $type);
        return $this->getAuditLogs($n, $type, function($query) use($logger) {
            return $query->forLogger($logger)->dateOrder();
        });
    }

    public function forLoggerPaginated($logger, $per_page = null, $type = null) {
        $this->setLoggerLoggeeAndTypes($logger, null, $type);
        return $this->getAuditLogsPaginated($per_page, $type, function($query) use($logger) {
            return $query->forLogger($logger)->dateOrder();
        });
    }

    public function forLoggee($loggee, $type = null, $n = null) {
        $this->setLoggerLoggeeAndTypes(null, $loggee, $type);
        return $this->getAuditLogs($n, $type, function($query) use($loggee) {
            return $query->forLoggee($loggee);
        });
    }

    public function forLoggeePaginated($loggee, $per_page = null, $type = null) {
        $this->setLoggerLoggeeAndTypes(null, $loggee, $type);
        return $this->getAuditLogsPaginated($per_page, $type, function($query) use($loggee) {
            return $query->forLoggee($loggee)->dateOrder();
        });
    }
    
    public function forLoggerAndLoggee($logger, $loggee, $type = null, $n = null, $inclusive = true) {
        $this->setLoggerLoggeeAndTypes($logger, $loggee, $type);
        return $this->getAuditLogs($n, $type, function($query) use($logger, $loggee, $inclusive) {
            return $query->forLoggerAndLoggee($logger, $loggee, $inclusive)->dateOrder();
        });
    }

    public function forLoggerAndLoggeePaginated($logger, $loggee, $per_page = null, $type = null, $inclusive = true) {
        $this->setLoggerLoggeeAndTypes($logger, $loggee, $type);
        return $this->getAuditLogsPaginated($per_page, $type, function($query) use($logger, $loggee, $inclusive) {
            return $query->forLoggerAndLoggee($logger, $loggee, $inclusive)->dateOrder();
        });
    }

    public function getAuditLogs($n = null, $type = null, $closure = null) {
        $clock = 'get_audit_logs_' . time();
        $query = $this->buildQuery($n, $type, $closure);
        $audit_logs = $query->orderBy('id','desc')->get();
        // Parse audit_logs for composite logs
        if($this->parseCompositesEnabled()) {
            $this->parseComposites($audit_logs);
        }
        $logs = $this->addLoggerLoggeeAndTypes($audit_logs);
        return $logs;
    }

    protected function getAuditLogsPaginated($per_page = null, $type = null, $closure = null, $n = null) {
        $clock = 'get_audit_logs_' . time();
        $query = $this->buildQuery($n, $type, $closure);
        if(!$per_page) {
            $per_page = config('auditing.per-page-default');
        }
        $audit_logs = $query->orderBy('id','desc')->paginate($per_page);
        if($this->parseCompositesEnabled()) {
            $this->parseComposites($audit_logs);
        }
        $logs = $this->addLoggerLoggeeAndTypes($audit_logs);
        return $logs;
    }

    protected function buildQuery($n = null, $type = null, $closure = null) {
        $query = $this->audit_log;
        if(is_callable($closure)) {
            $query = $closure($query);
        }
        if($n) {
            $query = $query->take($n);
        }
        if($type) {
            if (is_array($type)) {
                $query = $query->where(function($query) use ($type) {
                    foreach($type as $each_type) {
                        $query = $query->orWhere('type', $each_type);
                    }
                });
            }
            else {
                $query = $query->whereType($type);
            }
        }
        return $query;
    }

    protected function parseComposites(&$audit_logs) {
        $composites = Config::get('auditing.composites');
        for($i = 0; $i < count($audit_logs); $i++) {
            foreach($composites as $composite_type => $sub_types) {
                // Because we're looping through audit logs in reverse chronological order, it means we'll come across
                // the last sub-type in a composite before the first one, since they're entered in chronological order
                if($audit_logs[$i]->type === end($sub_types)) {
                    // This is the end of the composite audit log $composite, which means
                    // the next count($sub_types) should belong to that audit log
                    $next_relevant_logs = $audit_logs->slice($i, count($sub_types));
                    $this_types = array_reverse(array_pluck($next_relevant_logs->toArray(),'type'));
                    if($this_types !== $sub_types) {
                        // Something has gone wrong because logs aren't in database in correct order to match up with composites, database is corrupt
                        // Just ignore these for now and leave un-composited logs in the array
                    } else {
                        $composite = new CompositeAuditLog($next_relevant_logs);
                        $composite->type = $composite_type;
                        $composite->sortBy(function($item) {
                            return $item->id;
                        });
                        $audit_logs->splice($i,count($sub_types),[$composite]);
                    }
                }
            }
        }
    }

    protected function getLoggeeNameAttribute($loggee) {
        if($loggee->name) {
            return $loggee->name;
        } else {
            return '#' . $loggee->id;
        }
    }

    public function getTypes($id_list = null) {
        $logs = $this->audit_log;
        if($id_list) {
            if(is_numeric($id_list[0])) {
                $logs = $logs->whereIn('id',$id_list)->select('type')->distinct()->get();
            } else {
                $logs = $id_list;
            }
        } else {
            $logs = $logs->select('type')->distinct()->get();
        }
        $types = [];
        foreach($logs as $log) {
            if(!isset($types[$log->type])) {
                $types[$log->type] = $log->type_human_readable;
            }
        }
        return $types;
    }

    public function groupByInterval($logs, $interval_mins = 15) {
        $grouped_by_interval = [];
        foreach($logs as $log) {
            $timestamp = strtotime($log->created_at);
            $interval_secs = $interval_mins * 60;
            $timestamp_rounded = $timestamp - ($timestamp % $interval_secs);
            $group = Carbon::createFromTimestamp($timestamp_rounded);
            $grouped_by_interval[$group->format('Y-m-d H:i:s')][] = $log;
        }
        return $grouped_by_interval;
    }

    protected function parseCompositesEnabled() {
        if(Config::get('auditing.enable_composites') && Config::has('auditing.composites') && !empty(Config::get('auditing.composites'))) {
            return true;
        }
        return false;
    }

    protected function addLoggerLoggeeAndTypes($audit_logs) {
        $audit_logs->setLogger($this->logger);
        $audit_logs->setLoggee($this->loggee);
        $audit_logs->setTypes($this->types);
        return $audit_logs;
    }

}