<?php namespace Concore\Properties\Http\Controllers;

use Input, Redirect;
use Illuminate\Http\Request;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Collection;
use Concore\Properties\Models\Property;
use Concore\Properties\Services\PropertyService;

class PropertyController extends \Illuminate\Routing\Controller {

	protected $property;
	protected $property_service;

	public function __construct(Property $property, PropertyService $property_service) {
		$this->property = $property;
		$this->property_service = $property_service;
	}

	protected function parseInput() {
		$property_inputs = ['opportunity_types','property_types','max_size', 'min_size', 'location', 'distance','term'];
		$property_input_values = [];
		foreach($property_inputs as $property_input) {
			$property_input_value = \Input::get($property_input, null);
			if (!is_null($property_input_value)) {
				if(in_array($property_input,['max_size','min_size'])) {
					$property_input_value = str_replace(',','',$property_input_value);
				}
				$property_input_values[$property_input] = $property_input_value;
			}
		}
		return $property_input_values;
	}

	public function index(Request $request) {
		$filters = $this->parseInput();
		$properties = Collection::make();
		if(!$filters) {
			if($filters = $this->property_service->getFiltersFromSession()) {
				return Redirect::route('properties',$filters);
			} else {
				$properties = $this->property_service->getLatest();
			}
		} else {
			if (!\Session::has('errors')) {
				$properties = $this->property_service->searchForProperties($filters);
			}
		}

		$count = $properties->count();

		// Paginate
		$perPage = 10; // Item per page (change if needed)
		$currentPage = Input::has('page') ? Input::get('page') : 1;
		$paginator = new LengthAwarePaginator($properties, $count, $perPage, $currentPage);
		$paginator->setPath($request->url());
		if($filters) {
			foreach($filters as $key => $value) {
				$paginator->addQuery($key, $value);
			}
		}
		$properties = Collection::make($properties->slice(($currentPage-1) * $perPage, $perPage)->all());
		return view('properties::index')->with([
			'properties' => $properties,
			'input' => $filters,
			'paginator' => $paginator,
			'count' => $count
		]);
	}

	public function show($key) {
		$property = $this->property_service->getProperty($key);
		if(!$property) {
			return Redirect::route('properties',[],301);
		}
		return view('properties::show')->with('property',$property);
	}

	public function addToFavourites(Request $request, $id) {
        $this->property = $this->property->find($id);
		$this->property_service->addToFavourites($this->property);
		return $this->redirectBackIfNotAjax($request);
	}

	public function removeFromFavourites(Request $request, $id) {
		$this->property = $this->property->find($id);
		$this->property_service->removeFromFavourites($this->property);
		return $this->redirectBackIfNotAjax($request);
	}

	public function removeAllFavourites(Request $request) {
		$this->property_service->removeAllFavourites();
		return $this->redirectBackIfNotAjax($request);
	}

	public function viewFavourites() {
		$properties = $this->property_service->getFavouritesFromCookieIds();
		return view('properties::favourites')->with('properties', $properties);
	}

	protected function redirectBackIfNotAjax(Request $request) {
		if (!$request->ajax())
		{
		    return Redirect::route('property',['key' => $this->property->key]);
		}
	}

}
