<?php namespace Concore\Properties\Services;

use Cookie, Session, Config;
use Concore\Properties\Models\Property;

class PropertyService {

	protected $property;

	public function __construct(Property $property) {
		$this->property = $property;
	}

	public function getLatest($n = 10) {
		$properties = $this->property->orderBy('created_at','desc')->limit(0,$n)->get();
		$this->checkFavourites($properties);
		return $properties;
	}

	public function storeFiltersInSession($filters) {
		return Session::put('properties.filters',$filters);
	}

	public function getFiltersFromSession() {
		return Session::get('properties.filters');
	}

	public function searchForProperties($filters) {

		$properties = [];

		if(count($filters)) {

			$properties = $this->property->query();

			if(isset($filters['term'])) {

				$keywords = explode(' ', $filters['term']);

				$term = '';

				foreach($keywords as $keyword) {
					$term .= '%'.$keyword;
				}

				$term .= '%';

				$properties = $properties->where(function($query) use ($term) {
					$query->where('name', 'LIKE', $term)
					      ->orWhere('intro', 'LIKE', $term)
					      ->orWhere('body', 'LIKE', $term);
				});
			}

			if(isset($filters['opportunity_types'])) {
				$properties = $properties->whereHas('opportunity_types',function($q) use($filters) {
					$q->whereIn('opportunity_types.id',$filters['opportunity_types']);
				});
			}

			$land = false;

			if(isset($filters['property_types'])) {
				$properties = $properties->whereHas('property_types',function($q) use($filters) {
					$q->whereIn('property_types.id',$filters['property_types']);
				});

				foreach($filters['property_types'] as $property_type) {
					if (strpos(strtolower($property_type), 'land') !== false) {
						$land = true;
					}
				}
			}

			if(isset($filters['min_size'])) {
				if ($land) {
					$properties = $properties->where('land_max','>=',($filters['min_size']/43560));
				}
				else {
					$properties = $properties->where('floor_max','>=',$filters['min_size']);
				}
			}
			if(isset($filters['max_size'])) {
				if ($land) {
					$properties = $properties->where('land_min','<=',($filters['max_size']/43560));
				}
				else {
					$properties = $properties->where('floor_min','<=',$filters['max_size']);
				}
			}

			if(isset($filters['location']) && $filters['location'] !== '') {
				if (!isset($filters['distance']) || $filters['distance'] == '0') {
					$filters['distance'] = 999999;
				}
                $within_distance = $this->findPropertiesWithinDistance($filters['distance'], $filters['location']);

				$properties = $properties->whereIn('id', array_pluck($within_distance,'id'));

                $properties = $properties->get();

                $properties = $this->sortByDistance($properties, $within_distance);

			} else {
                $properties = $properties->get();
            }

		}

		$this->checkFavourites($properties);

		$this->storeFiltersInSession($filters);

		return $properties;
	}

	public function getFavouritesFromCookieIds() {
		$property_favourites = $this->getFavouriteIdsFromCookie();
		$properties = $this->property->find($property_favourites);
		$this->checkFavourites($properties);
		return $properties;
	}

	public function countFavourites() {
		$property_favourites = $this->getFavouriteIdsFromCookie();
		return count($property_favourites);
	}

	public function getProperty($key) {
		$property = $this->property->where('key',$key)->first();
		if($property) {
			$property_favourites = $this->getFavouriteIdsFromCookie();
			$this->checkFavourite($property, $property_favourites);
		}
		return $property;
	}

	public function addToFavourites(Property $property) {
		$property_favourites = $this->getFavouriteIdsFromCookie();
		$property_favourites[$property->id] = $property->id;
		$this->queueFavouriteCookie($property_favourites);
		return true;
	}

	public function removeFromFavourites(Property $property) {
		$property_favourites = $this->getFavouriteIdsFromCookie();
		if (isset($property_favourites[$property->id])) {
			unset($property_favourites[$property->id]);
		}
		$this->queueFavouriteCookie($property_favourites);
		return true;
	}

	public function removeAllFavourites() {
		Cookie::queue(Cookie::forget('property_favourites'));
	}

	public function getFavouriteIdsFromCookie() {
		return Cookie::get('property_favourites', array());
	}

	private function checkFavourite(Property &$property, $property_favourites) {
		if (in_array($property->id, $property_favourites)) {
			$property->is_favourite = true;
			return true;
		}
		return false;
	}

	private function checkFavourites(&$properties) {
		if ($properties) {
			$property_favourites = $this->getFavouriteIdsFromCookie();
			foreach($properties as $property) {
				$this->checkFavourite($property, $property_favourites);
			}
		}
	}

	private function queueFavouriteCookie($property_favourites) {
		Cookie::queue('property_favourites', $property_favourites, 262974);
	}

	private function findPropertiesWithinDistance($distance, $location, $units = 'm') {
		$geo_service = app('Concore\Geo\Services\GeoService');
		$within_distance = [];
		$loc_latlng = $geo_service->getGeocodeLatlng($location);
		$properties = $this->property->with(['addresses'=>function($query) {
			$query->select('addresses.latitude','addresses.longitude');
		}])->select('properties.id')->has('addresses')->get();
		foreach($properties as $property) {
			$property_distance = $geo_service->getDistance($loc_latlng, $property->address->latlng, ['units'=>$units]);
			if ($property_distance <= $distance) {
				$within_distance[$property->id] = [
                    'id' => $property->id,
                    'distance' => $property_distance
                ];
			}
		}

        return $within_distance;
	}

    private function sortByDistance($properties, $ids_and_distances) {
        return $properties->sort(function($a, $b) use ($ids_and_distances) {
            return $ids_and_distances[$a->id]['distance'] > $ids_and_distances[$b->id]['distance'];
        });
    }

}