<?php namespace Concore\Sam\Reports;

use DB, Carbon\Carbon;
use Concore\Foundation\Models\Traits\IdentifiableTrait;
use Concore\Reports\Traits\ReportableTrait;
use Concore\Reports\Table;
use Concore\Sam\Reports\Interfaces\SamReportableInterface;
use Concore\Sam\Reports\Traits\SamReportableTrait;
use Concore\Sam\Models\Task;
use Concore\Personnel\Models\Nature;
use Concore\Personnel\Models\Timeslot;

class ActivitiesReport implements SamReportableInterface {

    use IdentifiableTrait;
    use ReportableTrait {
            getDates as getDatesTrait;
            ReportableTrait::getUrl insteadof IdentifiableTrait;
        }
    use SamReportableTrait;

    private $task;
    private $nature;
    private $timeslot;

    public function __construct(Task $task, Nature $nature, Timeslot $timeslot) {
        $this->task = $task;
        $this->nature = $nature;
        $this->timeslot = $timeslot;
    }

    public function getNameAttribute() {
        return 'Activities';
    }

    public function getDates($query = null, $field = null, $format = null) {
        $query = $this->task;
        if (!is_null($this->organisation_id)) {
            $query = $this->getOrganisationTasksQuery($query);
        }
        return $this->getDatesTrait($query,'date','Y-m-d');
    }

    public function getTables() {
        if (!$this->tables) {
            $tables['activities_totals_table'] = $this->getTotalsTable();
            $tables['individual_activity_totals_table'] = $this->getIndividualActivityTotalsTable();
            $tables['timeslot_totals_table'] = $this->getTimeslotTotalsTable();
            $this->tables = $tables;
        }
        return $this->tables;
    }

    protected function getFirstDate($query, $field, $format) {
        return $this->first = Carbon::createFromFormat('Y-m-d', '2012-01-01');
    }

    protected function getLastDate($query, $field, $format) {
        return $this->last = Carbon::now();
    }

    public function getTotalsTable() {
        $tasks = $this->task->selectRaw('COUNT(*) as count, sum(duration) as time, sum(duration) / COUNT(*) as average, COUNT(DISTINCT service_user_case_id) as total_cases, sum(duration) / COUNT(DISTINCT service_user_case_id) as average_per_case')
            ->join('service_user_cases','service_user_cases.id','=','tasks.service_user_case_id');
        if($this->organisation_id) {
            $tasks = $tasks->join('service_users','service_users.id','=','service_user_cases.id')
                ->join('people','people.id','=','service_users.person_id')
                ->where('people.organisation_id','=',$this->organisation_id);
        }
        $tasks = $tasks->where('tasks.date','>=',$this->from)
            ->first()
            ->toArray();
        return new Table(
            'Totals',
            [],
            [
                [
                    'Total number',
                    $tasks['count']
                ],
                [
                    'Total duration',
                    $tasks['time']
                ],
                [
                    'Average duration',
                    $tasks['average']
                ],
                [
                    'Total affected cases',
                    $tasks['total_cases']
                ],
                [
                    'Average duration per case',
                    $tasks['average_per_case']
                ]
            ],
            [
                1 => [
                    1 => $this->formatDuration()
                ],
                2 => [
                    1 => $this->formatDuration()
                ],
                4 => [
                    1 => $this->formatDuration()
                ]
            ]
        );
    }

    public function getIndividualActivityTotalsTable() {
        \DB::enableQueryLog();
        // Left join to get activities in date range, even if they have no nature, including counts and averages
        $activities_left = \DB::table('tasks')
            ->selectRaw('
                CASE WHEN tasks.nature_other IS NOT NULL THEN "other" ELSE IFNULL(natures.id,"undefined") END as id,
                CASE WHEN tasks.nature_other IS NOT NULL THEN "Other" ELSE IFNULL(natures.name,"Undefined") END as nature,
                COUNT(tasks.id) as count,
                sum(duration) as time,
                sum(duration) / COUNT(*) as average,
                COUNT(DISTINCT(tasks.service_user_case_id)) as service_user_cases
            ')
            ->where('tasks.date','>=',$this->from)
            ->where('tasks.date','<=',$this->to)
            ->join('natures','natures.id','=','tasks.nature_id','left outer')
            ->groupBy('nature');
        if($this->organisation_id) {
            $activities_left = $activities_left->join('service_user_cases','service_user_cases.id','=','tasks.service_user_case_id')
                ->join('service_users','service_users.id','=','service_user_cases.id')
                ->join('people','people.id','=','service_users.person_id')
                ->where('people.organisation_id','=',$this->organisation_id);
        }
        // Right join to get natures that have no activities, no need to get counts or averages as there will be no activities
        $activities_right = \DB::table('tasks')
            ->selectRaw('
                natures.id,
                natures.name as nature,
                0 as count,
                0 as time,
                0 as average,
                0 as service_user_cases
            ')
            ->whereNull('tasks.id')
            ->join('natures','natures.id','=','tasks.nature_id','right outer')
            ->groupBy('nature');
        // Get a union of these two (simulating FULL OUTER JOIN which MySQL can't do)
        // to get all activities with no nature and all natures with no activities, including counts etc
        $activities = $activities_left->union($activities_right)->get();

        if(count(array_filter(array_pluck($activities,'count')))) {
            $nature_ids = [];
            foreach($activities as &$activity) {
                $nature_ids[] = $activity->id;
                $activity = get_object_vars($activity);
                unset($activity['id']);
            }
        } else {
            $activities = $nature_ids = [];
        }

        return new Table(
            'Activities by nature',
            [
                '',
                'Number of activities',
                'Total activity duration',
                'Average activity duration',
                'Affected Cases'
            ],
            $activities,
            [
                '*' => [
                    2 => $this->formatDuration(),
                    3 => $this->formatDuration()
                ]
            ],
            function($row_key) use($nature_ids) {
                return route('sculpt.index',[
                    'activities',
                    'nature' => $nature_ids[$row_key],
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'organisation_id' => $this->organisation_id
                ]);
            }
        );
    }

    public function getTimeslotTotalsTable() {
        $timeslots = $this->task->selectRaw('timeslots.type,
                                             COUNT(*) as count,
                                             sum(tasks.duration) as time,
                                             sum(tasks.duration) / COUNT(*) as average,
                                             sum(timeslots.duration) as timeslot_time,
                                             sum(timeslots.duration) / COUNT(*) as timeslot_average,
                                             COUNT(DISTINCT(tasks.service_user_case_id)) as service_user_cases')
            ->join('timeslots','timeslots.task_id','=','tasks.id')
            ->where('tasks.date','>=',$this->from)
            ->where('tasks.date','<=',$this->to);
        if($this->organisation_id) {
            $timeslots = $timeslots->join('service_user_cases','service_user_cases.id','=','tasks.service_user_case_id')
                ->join('service_users','service_users.id','=','service_user_cases.id')
                ->join('people','people.id','=','service_users.person_id')
                ->where('people.organisation_id','=',$this->organisation_id);
        }
        $timeslots = $timeslots->groupBy('timeslots.type')->get()->toArray();
        $types = array_pluck($timeslots,'type');
        if(count($timeslots)) {
            foreach($this->timeslot->getTImeslotTypes() as $type) {
                if(!in_array($type, $types)) {
                    $timeslots[] = [
                        'type' => $type,
                        'count' => 0,
                        'time' => 0,
                        'average' => 0,
                        'service_user_cases' => 0
                    ];
                }
            }
        }
        return new Table(
            'Activities by what they involved',
            [
                '',
                'Number of Activities',
                'Total activity duration',
                'Average activity duration',
                'Total timeslot duration',
                'Average timeslot duration',
                'Affected Cases'
            ],
            $timeslots,
            [
                '*' => [
                    2 => $this->formatDuration(),
                    3 => $this->formatDuration(),
                    4 => $this->formatDuration(),
                    5 => $this->formatDuration()
                ]
            ],
            function($row_key) {
                return route('sculpt.index',[
                    'activities',
                    'timeslot_type' => (string)$row_key,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'organisation_id' => $this->organisation_id
                ]);
            }
        );
    }

    private function getOrganisationTimeslotsQuery($query) {
        return $query->whereHas('task', function ($task) {
            $this->getOrganisationTasksQuery($task);
        });
    }

    private function getOrganisationTasksQuery($query) {
        return $query->whereHas('user', function ($user) {
            $user->whereHas('employee', function ($employee) {
                $employee->whereHas('person', function ($person) {
                    $person->whereHas('organisation', function ($organisation) {
                        $organisation->where('id', $this->organisation_id);
                    });
                });
            });
        });
    }

    private function getFromToTasksQuery($query) {
        return $query->where('date','>=',$this->from)->where('date','<=',$this->to);
    }

    private function getFromToTimeslotsQuery($query) {
        return $query->whereHas('task', function ($task) {
            $this->getFromToTasksQuery($task);
        });
    }

    private function formatDuration() {
        return function($input)
        {
            settype($input, 'integer');
            $mins = $input % 60;
            $hours = ($input - $mins) / 60;
            return sprintf('%d:%s', $hours, str_pad($mins,2,'0'));
        };
    }

}