<?php namespace Concore\Sam\Reports;

use DB, Carbon\Carbon;
use Concore\Foundation\Models\Traits\IdentifiableTrait;
use Concore\Reports\Traits\ReportableTrait;
use Concore\Reports\Table;
use Concore\Sam\Reports\Interfaces\SamReportableInterface;
use Concore\Sam\Reports\Traits\SamReportableTrait;
use Concore\Sam\Models\ServiceUser;
use Concore\Sam\Models\ServiceUserCase;
use Concore\Sam\Models\ActivePeriod;

class ServiceUserCasesReport extends MasterRoleReport
{

    private $service_user;
    private $service_user_case;
    private $active_period;

    public function __construct(
        ServiceUser $service_user,
        ServiceUserCase $service_user_case,
        ActivePeriod $active_period
    ) {
        $this->service_user = $service_user;
        $this->service_user_case = $service_user_case;
        $this->active_period = $active_period;
    }

    protected $role_name = 'service-user-case';

    protected $start_during_end_wording = [
        [
            'Open cases',
            'Cases yet to be opened',
            'Closed cases'
        ],
        [
            'First time case openings',
            'Case reopenings',
            'Case closings'
        ],
        [
            'Open cases',
            'Cases yet to be opened',
            'Closed cases'
        ]
    ];

    public function getNameAttribute() {
        return 'Service User Cases';
    }

    public function getDates($query = null, $field = null, $format = null) {
        $query = $this->active_period->has('service_user_case');
        if (!is_null($this->organisation_id)) {
            $query = $query->whereHas('service_user_case',function($query) {
                $query->inOrganisation($this->organisation_id);
            });
        }
        return $this->getDatesTrait($query,['start','end'],'Y-m-d');
    }

    protected function getFirstDate($query, $field, $format) {
        return $this->first = Carbon::createFromFormat('Y-m-d', '2012-01-01');
    }

    protected function getLastDate($query, $field, $format) {
        return $this->last = Carbon::now();

//        $count_query = clone $query;
//        if ($count_query->select('id')->whereNull('end')->count()) {
//            $this->last = Carbon::now();
//        }
//        else {
//            $end_query = clone $query;
//            $last = $end_query->select('end')->max('end');
//            if (!$last) {
//                $last = $query->select('start')->max('start');
//            }
//            $this->last = $last ? Carbon::createFromFormat($format, $last) : $this->first;
//        }
//        return $this->last;
    }

    public function getTables() {
        if (!$this->tables) {
            $tables['start_of_period_table'] = $this->getStartOfPeriodTable();
            $tables['during_period_table'] = $this->getDuringPeriodTable();
            $tables['end_of_period_table'] = $this->getEndOfPeriodTable();
            $tables['case_types_table'] = $this->getCaseTypesTable();
            $tables['activities_table'] = $this->getActivitiesTable();
            $tables['gender_totals_table'] = $this->getGenderTotalsTable();
            $tables['referral_totals_table'] = $this->getReferralTotalsTable();
            $tables['referral_methods_totals_table'] = $this->getReferralMethodsTotalsTable();
            $tables['how_they_heard_totals_table'] = $this->getHowTheyHeardTotalsTable();
            $tables['ethnicity_totals_table'] = $this->getEthnicityTotalsTable();
            $tables['age_totals_table'] = $this->getAgeTotalsTable();
            $this->tables = $tables;
        }
        return $this->tables;
    }

    protected function getBaseQuery() {
        $query = $this->service_user_case;
        if (!is_null($this->organisation_id)) {
            $query = $query->inOrganisation($this->organisation_id);
        }
        return $query;
    }

    protected function getActivePeriodBaseQuery() {
        $query = $this->active_period;
        if (!is_null($this->organisation_id)) {
            $query = $query->whereHas('service_user_case', function($query) {
                $query->inOrganisation($this->organisation_id);
            });
        } else {
            $query = $query->has('service_user_case');
        }
        return $query;
    }

    public function getCaseTypesTable() {
        if (is_null($this->from)) {
            if (is_null($this->first)) {
                $this->getDates();
            }
            $from = $this->first;
        }
        else {
            $from = $this->from;
        }
        if (is_null($this->to)) {
            if (is_null($this->last)) {
                $this->getDates();
            }
            $to = $this->last;
        }
        else {
            $to = $this->to;
        }

        $query = $this->getBaseQuery();

        $counts = $query->selectRaw('case_type, COUNT(*) as count')->whereHas('active_periods', function($active_period) use ($from, $to) {
            $active_period->where(function($q) use ($from, $to) {
                $q->where('start','>=',$from->format('Y-m-d'))->where('start','<=',$to->format('Y-m-d'));
            });
        })->selectRaw('case_type')->groupBy('case_type')->get()->toArray();

        return new Table(
            'Case type',
            [
                '',
                'Quantity'
            ],
            $counts
        );
    }

    public function getActivitiesTable() {
        if (is_null($this->from)) {
            if (is_null($this->first)) {
                $this->getDates();
            }
            $from = $this->first;
        }
        else {
            $from = $this->from;
        }
        if (is_null($this->to)) {
            if (is_null($this->last)) {
                $this->getDates();
            }
            $to = $this->last;
        }
        else {
            $to = $this->to;
        }

        $case_ids = false;

        if (!is_null($this->organisation_id)) {
            $query = $this->getBaseQuery();
            $case_ids = $query->lists('id');
        }

        $activities_count = DB::table('tasks')
            ->select(DB::raw('count(*) as count'))
            ->join('service_user_cases', 'tasks.service_user_case_id', '=', 'service_user_cases.id')
            ->where('tasks.date','>=',$from->format('Y-m-d'))
            ->where('tasks.date','<=',$to->format('Y-m-d'))
            ->groupBy('service_user_cases.id');

        if ($case_ids) {
            $activities_count = $activities_count->whereIn('service_user_cases.id', $case_ids);
        }

        $average_activities_count = DB::table(DB::raw('('.$activities_count->toSql().') as tasks'))
            ->select(DB::raw('avg(tasks.count) as average'))
            ->mergeBindings($activities_count)
            ->first();

        $activities_time = DB::table('tasks')
            ->select(DB::raw('sum(duration) as time'))
            ->join('service_user_cases', 'tasks.service_user_case_id', '=', 'service_user_cases.id')
            ->where('tasks.date','>=',$from->format('Y-m-d'))
            ->where('tasks.date','<=',$to->format('Y-m-d'))
            ->groupBy('service_user_cases.id');

        if ($case_ids) {
            $activities_time = $activities_time->whereIn('service_user_cases.id', $case_ids);
        }

        $average_activities_time = DB::table(DB::raw('('.$activities_time->toSql().') as tasks'))
            ->select(DB::raw('avg(tasks.time) as average'))
            ->mergeBindings($activities_time)
            ->first();

        return new Table(
            'Activities',
            [],
            [
                [
                    'Average number of activities received during this period*',
                    round($average_activities_count->average, 1)
                ],
                [
                    "Average total hours' support received during this period*",
                    (int) $average_activities_time->average
                ]
            ]
        );
    }

    public function getGenderTotalsTable() {
        if (is_null($this->from)) {
            if (is_null($this->first)) {
                $this->getDates();
            }
            $from = $this->first;
        }
        else {
            $from = $this->from;
        }
        if (is_null($this->to)) {
            if (is_null($this->last)) {
                $this->getDates();
            }
            $to = $this->last;
        }
        else {
            $to = $this->to;
        }
        $query = $this->getBaseQuery();
        $gender_counts = $query->selectRaw('IFNULL(genders.id,"undefined") as id, IFNULL(genders.name,"Undefined") as gender, count(*) as quantity')->whereHas('active_periods', function($active_period) use ($from, $to) {
            $active_period->where(function($q) use ($from, $to) {
                $q->where('start','>=',$from->format('Y-m-d'))->where('start','<=',$to->format('Y-m-d'));
            });
        })
            ->join('service_users', 'service_user_cases.service_user_id', '=', 'service_users.id')
            ->join('people', 'service_users.person_id', '=', 'people.id')
            ->leftJoin('genders', 'people.gender_id', '=', 'genders.id')
            ->groupBy('genders.name')
            ->get();
        $gender_counts = $gender_counts->toArray();

        $gender_ids = array_pluck($gender_counts,'id');
        foreach($gender_counts as &$value) {
            unset($value['id']);
        }

        return new Table(
            'Gender',
            [
                '',
                'Quantity'
            ],
            $gender_counts,
            null,
            function($row_index) use($gender_ids) {
                return route('sculpt.index',[
                    'service-user-cases',
                    'organisation_id' => $this->organisation_id,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'gender' => $gender_ids[$row_index]
                ]);
            }
        );
    }

    public function getReferralTotalsTable() {
        if (is_null($this->from)) {
            if (is_null($this->first)) {
                $this->getDates();
            }
            $from = $this->first;
        }
        else {
            $from = $this->from;
        }
        if (is_null($this->to)) {
            if (is_null($this->last)) {
                $this->getDates();
            }
            $to = $this->last;
        }
        else {
            $to = $this->to;
        }
        $query = $this->getBaseQuery();
        $referral_counts = $query->selectRaw('
            IFNULL(referral_sources.id,
                (CASE WHEN service_user_cases.referral_source_other IS NOT NULL THEN "other" ELSE "undefined" END)
            ) as id,
            IFNULL(referral_sources.name,
                (CASE WHEN service_user_cases.referral_source_other IS NOT NULL THEN "Other" ELSE "Undefined" END)
            ) as referral_source,
            count(*) as quantity
        ')->whereHas('active_periods', function($active_period) use ($from, $to) {
            $active_period->where(function($q) use ($from, $to) {
                $q->where('start','>=',$from->format('Y-m-d'))->where('start','<=',$to->format('Y-m-d'));
            });
        })
            ->leftJoin('referral_sources', 'service_user_cases.referral_source_id', '=', 'referral_sources.id')
            ->groupBy('referral_source')
            ->get()->toArray();
        $referral_ids = array_pluck($referral_counts,'id');
        foreach($referral_counts as &$value) {
            unset($value['id']);
        }

        return new Table(
            'Referral source',
            [
                '',
                'Quantity'
            ],
            $referral_counts,
            null,
            function($row_index) use($referral_ids) {
                return route('sculpt.index',[
                    'service-user-cases',
                    'organisation_id' => $this->organisation_id,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'referral_source' => $referral_ids[$row_index]
                ]);
            }
        );
    }

    public function getReferralMethodsTotalsTable() {
        if (is_null($this->from)) {
            if (is_null($this->first)) {
                $this->getDates();
            }
            $from = $this->first;
        }
        else {
            $from = $this->from;
        }
        if (is_null($this->to)) {
            if (is_null($this->last)) {
                $this->getDates();
            }
            $to = $this->last;
        }
        else {
            $to = $this->to;
        }
        $query = $this->getBaseQuery();
        $referral_counts = $query->selectRaw('
            IFNULL(referral_methods.id,
                (CASE WHEN service_user_cases.referral_method_other IS NOT NULL THEN "other" ELSE "undefined" END)
            ) as id,
            IFNULL(referral_methods.name,
                (CASE WHEN service_user_cases.referral_method_other IS NOT NULL THEN "Other" ELSE "Undefined" END)
            ) as referral_method,
            count(*) as quantity
        ')->whereHas('active_periods', function($active_period) use ($from, $to) {
            $active_period->where(function($q) use ($from, $to) {
                $q->where('start','>=',$from->format('Y-m-d'))->where('start','<=',$to->format('Y-m-d'));
            });
        })
            ->leftJoin('referral_methods', 'service_user_cases.referral_method_id', '=', 'referral_methods.id')
            ->groupBy('referral_method')
            ->get()->toArray();
        $referral_ids = array_pluck($referral_counts,'id');
        foreach($referral_counts as &$value) {
            unset($value['id']);
        }

        return new Table(
            'Referral method',
            [
                '',
                'Quantity'
            ],
            $referral_counts,
            null,
            function($row_index) use($referral_ids) {
                return route('sculpt.index',[
                    'service-user-cases',
                    'organisation_id' => $this->organisation_id,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'referral_method' => $referral_ids[$row_index]
                ]);
            }
        );
    }

    public function getHowTheyHeardTotalsTable() {
        if (is_null($this->from)) {
            if (is_null($this->first)) {
                $this->getDates();
            }
            $from = $this->first;
        }
        else {
            $from = $this->from;
        }
        if (is_null($this->to)) {
            if (is_null($this->last)) {
                $this->getDates();
            }
            $to = $this->last;
        }
        else {
            $to = $this->to;
        }
        $query = $this->getBaseQuery();

        $how_they_heard_counts = $query->selectRaw('
            IFNULL(how_they_heard,
                (CASE WHEN how_they_heard_other IS NOT NULL THEN "Other" ELSE "Undefined" END)
            ) as _how_they_heard,
            count(*) as quantity
        ')->whereHas('active_periods', function($active_period) use ($from, $to) {
            $active_period->where(function($q) use ($from, $to) {
                $q->where('start','>=',$from->format('Y-m-d'))->where('start','<=',$to->format('Y-m-d'));
            });
        })
            ->groupBy('_how_they_heard')
            ->get()->toArray();

        return new Table(
            'How the service user heard about the project',
            [
                '',
                'Quantity'
            ],
            $how_they_heard_counts,
            null,
            function($row_index, $row) {
                return route('sculpt.index',[
                    'service-user-cases',
                    'organisation_id' => $this->organisation_id,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'how_they_heard' => $row['_how_they_heard'] !== 'Undefined' ? $row['_how_they_heard'] : 'Undefined',
                ]);
            }
        );
    }

    public function getEthnicityTotalsTable() {
        if (is_null($this->from)) {
            if (is_null($this->first)) {
                $this->getDates();
            }
            $from = $this->first;
        }
        else {
            $from = $this->from;
        }
        if (is_null($this->to)) {
            if (is_null($this->last)) {
                $this->getDates();
            }
            $to = $this->last;
        }
        else {
            $to = $this->to;
        }
        $query = $this->getBaseQuery();
        $ethnicity_counts = $query->selectRaw('IFNULL(people.ethnicity,\'Undefined\') as ethnicity, count(*) as quantity')->whereHas('active_periods', function($active_period) use ($from, $to) {
            $active_period->where(function($q) use ($from, $to) {
                $q->where('start','>=',$from->format('Y-m-d'))->where('start','<=',$to->format('Y-m-d'));
            });
        })
            ->join('service_users', 'service_user_cases.service_user_id', '=', 'service_users.id')
            ->join('people', 'service_users.person_id', '=', 'people.id')
            ->groupBy('people.ethnicity');
        $ethnicity_counts = $ethnicity_counts->get();
        $ethnicity_counts = $ethnicity_counts->toArray();
        return new Table(
            'Ethnicity',
            [
                '',
                'Quantity'
            ],
            $ethnicity_counts,
            null,
            function($row_index, $row) {
                return route('sculpt.index',[
                    'service-user-cases',
                    'organisation_id' => $this->organisation_id,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'ethnicity' => $row['ethnicity'] !== 'Undefined' ? $row['ethnicity'] : 'undefined',
                ]);
            }
        );
    }

    public function getAgeTotalsTable() {
        if (is_null($this->from)) {
            if (is_null($this->first)) {
                $this->getDates();
            }
            $from = $this->first;
        }
        else {
            $from = $this->from;
        }
        if (is_null($this->to)) {
            if (is_null($this->last)) {
                $this->getDates();
            }
            $to = $this->last;
        }
        else {
            $to = $this->to;
        }
        $query = $this->getBaseQuery();
        $ages_grouped = $query->selectAgeGroups()->whereHas('active_periods', function($active_period) use ($from, $to) {
            $active_period->where(function($q) use ($from, $to) {
                $q->where('start','>=',$from->format('Y-m-d'))->where('start','<=',$to->format('Y-m-d'));
            });
        })->first()->toArray();
        $age_counts = [];
        if(count(array_filter($ages_grouped))) {
            foreach($ages_grouped as $grouping => $count) {
                $age_counts[] = [
                    'group' => $grouping,
                    'count' => $count
                ];
            }
        }
        return new Table(
            'Age',
            [
                '',
                'Quantity'
            ],
            $age_counts,
            null,
            function($row_index, $row) {
                return route('sculpt.index',[
                    'service-user-cases',
                    'organisation_id' => $this->organisation_id,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'age' => $row['group'] !== 'Undefined' ? $row['group'] : 'undefined',
                ]);
            }
        );
    }

}