<?php namespace Concore\Sam\Services\Sculpt;

use Session, AuditLog, Auth, Carbon\Carbon;
use Concore\Sculpt\Services\Interfaces\SculptModelServiceInterface;
use Concore\Sculpt\Services\SculptModelService;

class ServiceUserCaseService extends SculptModelService implements SculptModelServiceInterface {

    public function queryIndex() {
        // Get in alphabetical order by person first name
        $query = $this->model->query();//->select(['service_user_cases.*']);//->join('service_users','service_user_cases.service_user_id','=','service_users.id')->join('people','service_users.person_id','=','people.id')->orderBy('people.first_name','asc');
        $user = Auth::user();
        if(!$user->can('see:all-organisations')) {
            $query = $query->inOrganisation($user->getOrganisationId());
        }
        if($user->hasRole('advocate') && $user->can('see:own-cases') && !$user->can('see:all-cases')) {
            $query = $query->forAdvocate($user->employee->advocate->id);
        }
        if($input = Session::get('forms.search.service-user-cases')) {
            if($this->inputHas($input,'id')) {
                $query->where('service_user_cases.id','=',$input['id']);
            }
            if($this->inputHas($input,'name')) {
                $query->whereHas('service_user',function($query) use($input) {
                    $query->whereHas('person',function($query) use($input) {
                        $terms = explode(' ',$input['name']);
                        foreach($terms as $term) {
                            $query->where(function ($query) use ($input, $term) {
                                $query->where('first_name', 'LIKE', '%' . $term . '%');
                                $query->orWhere('preferred_name', 'LIKE', '%' . $term . '%');
                                $query->orWhere('last_name', 'LIKE', '%' . $term . '%');
                            });
                        }
                    });
                });
            }
            if ($this->inputHas($input,['organisation_id','gender','ethnicity','age'])) {
                $query->whereHas('service_user', function ($query) use ($input, $user) {
                    $query->whereHas('person', function ($query) use ($input, $user) {
                        if($user->can('see:all-organisations') && $this->inputHas($input,'organisation_id')) {
                            $query->where('organisation_id', '=', $input['organisation_id']);
                        }
                        if($this->inputHas($input,'gender')) {
                            if($input['gender'] === 'undefined') {
                                $query->whereNull('gender_id');
                            } else {
                                $query->where('gender_id', '=', $input['gender']);
                            }
                        }
                        if($this->inputHas($input,'ethnicity')) {
                            if($input['ethnicity'] === 'undefined') {
                                $query->whereNull('ethnicity');
                            } else {
                                $query->where('ethnicity', '=', $input['ethnicity']);
                            }
                        }
                        if($this->inputHas($input,'age')) {
                            if($input['age'] === 'undefined') {
                                $query->where(function($query) {
                                    $query->whereNull('age')->orWhere('age','=','0');
                                });
                            } else {
                                list($from,$to) = explode('-',$input['age']);
                                $query->where(function($query) use($from,$to) {
                                    $query->where('age','>=',$from)->where('age','<=',$to);
                                });
                            }
                        }
                    });
                });
            }
            if($this->inputHas($input,['date_from','date_to'])) {
                $date_from = empty($input['date_from']) ? null : Carbon::createFromFormat('Y-m-d',$input['date_from'].'-01');
                $date_to = empty($input['date_to']) ? null : Carbon::createFromFormat('Y-m-d',$input['date_to'].'-01');
                $query->whereHas('active_periods',function($query) use($date_from, $date_to) {
                    if($date_from) {
                        $query->where('start','>=',$date_from->format('Y-m-d'));
                    }
                    if($date_to) {
                        $query->where('start','<=',$date_to->format('Y-m-t'));
                    }
                });
            }
            if($this->inputHas($input,'referral_method')) {
                if($input['referral_method'] === 'undefined') {
                    $query->where(function($query) {
                        $query->whereNull('referral_method_id')
                            ->whereNull('referral_method_other');
                    });
                } else if($input['referral_method'] === 'other') {
                    $query->whereNotNull('referral_method_other');
                } else {
                    $query->where('referral_method_id',$input['referral_method']);
                }
            }
            if($this->inputHas($input,'referral_source')) {
                if($input['referral_source'] === 'undefined') {
                    $query->where(function($query) {
                        $query->whereNull('referral_source_id')
                            ->whereNull('referral_source_other');
                    });
                } else if($input['referral_source'] === 'other') {
                    $query->whereNotNull('referral_source_other');
                } else {
                    $query->where('referral_source_id',$input['referral_source']);
                }
            }
            if($this->inputHas($input,'how_they_heard')) {
                if($input['how_they_heard'] === 'Undefined') {
                    $query->where(function($query) {
                        $query->whereNull('how_they_heard')
                              ->whereNull('how_they_heard_other');
                    });
                } else if($input['how_they_heard'] === 'Other') {
                    $query->whereNotNull('how_they_heard_other');
                } else {
                    $query->where('how_they_heard',$input['how_they_heard']);
                }
            }
            if(!empty($input['open-cases-only']) && (bool)$input['open-cases-only']) {
                $query->open();
            }
        }
        return $query;
    }

    protected function saveRecord($record, $input)
    {

        $contact_entities = null;
        if(isset($input['contact-entities'])) {
            $contact_entities = $input['contact-entities'];
            unset($input['contact-entities']);
        }

        // Save and attach relations before saving record
        $this->saveRelations($record, ["cancer_types","advocacy_issues","advocacy_outcomes","service_user","active_periods","contact_entities"], $input);

        // Save record
        if (isset($input[$this->model->getSlugged()])) {
            $record->fill($this->parseInput($input[$this->model->getSlugged()]));
        }
        $record->save();

        $this->saveRelations($record, ["notes","advocates"], $input);

        if($contact_entities) {
            $record->contact_entities()->detach();
            $contact_entities = array_filter($contact_entities);
            foreach($contact_entities as $contact_entity) {
                $contact_entity_type_id = isset($contact_entity['_pivot']['contact_entity_type_id']) && !empty($contact_entity['_pivot']['contact_entity_type_id']) ? $contact_entity['_pivot']['contact_entity_type_id'] : null;
                if(isset($contact_entity['contact-entity']['id']) && !empty($contact_entity['contact-entity']['id'])) {
                    $record->contact_entities()->attach($contact_entity['contact-entity']['id'],['contact_entity_type_id' => $contact_entity_type_id]);
                } else {
                    // Make new one and attach
                    $related_model = $record->contact_entities()->getRelated();
                    $service = $this->getNewServiceInstance($related_model);
                    $service->setModel($related_model);
                    $model = $service->createRecord($contact_entity);
                    $model->save();
                    $record->contact_entities()->attach($model->id,['contact_entity_type_id' => $contact_entity_type_id]);
                }
            }
            $record->save();
        }

        return $record;
    }

    public function updateRecord($id, $input) {
        $this->setRecord($id);
        $case = $this->queryUpdate($id)->first();

        if(Auth::user()->can('do:change-case-advocates') && isset($input['advocates'])) {

            $advocates_already_present = [];
            $advocates_removed_names = [];
            $advocates_added_names = [];

            if ($case->advocates) {
                foreach ($case->advocates as $advocate) {
                    if (!isset($input['advocates']) || $input['advocates'] === 'none' || !in_array($advocate->id, $input['advocates'])) {
                        $advocates_removed_names[] = $advocate->name;
                        AuditLog::put('removed-advocate', $advocate, ['case' => $case->name]);
                    } else {
                        $advocates_already_present[] = $advocate->id;
                    }
                }
            }

            if (isset($input['advocates'])) {
                $added_ids = [];
                if ($input['advocates'] !== 'none') {
                    foreach ($input['advocates'] as $input_advocate) {
                        if (!in_array($input_advocate, $advocates_already_present)) {
                            $added_ids[] = $input_advocate;
                        }
                    }
                }
                if ($added_ids) {
                    $advocate_model = app('Concore\Sam\Models\Advocate');
                    $added_advocates = $advocate_model->whereIn('id', $added_ids)->get();
                    foreach ($added_advocates as $added_advocate) {
                        $advocates_added_names[] = $added_advocate->name;
                        AuditLog::put('added-advocate', $added_advocate, ['case' => $case->name]);
                    }
                }
            }

            if ($advocates_removed_names || $advocates_added_names) {
                AuditLog::put('changed-advocates', $case, ['added' => $advocates_added_names, 'removed' => $advocates_removed_names]);
            }

        }

        $notes_already_present = [];

        if ($case->notes) {
            foreach($case->notes as $note) {
                $notes_already_present[] = $note->id;
            }
        }

        // Save record
        $case = $this->saveRecord($case, $input);

        $new_notes = $case->notes()->whereNotIn('note_id',$notes_already_present)->get();

        if ($new_notes) {
            foreach($new_notes as $new_note) {
                AuditLog::put('added-note',$case);
            }
        }

        return $case;
    }

    public function getRecentlyAdded($n = 5) {
        if(!$this->model) {
            $this->model = app('Concore\Sam\Models\ServiceUserCase');
        }
        if(!Auth::user()->can('see:all-cases')) {
            return [];
        }
        $query = $this->model->query();
        if(!Auth::user()->can('see:all-organisations')) {
            $query = $query->inOrganisation(Auth::user()->getOrganisationId());
        }
        $query = $query->orderBy('created_at','desc');
        return $query->take($n)->get();
    }

    public function getWithoutAdvocate() {
        if(!$this->model) {
            $this->model = app('Concore\Sam\Models\ServiceUserCase');
        }
        $query = $this->model->openOrYetToBeOpened()->withNoAdvocate();
        if(!Auth::user()->can('see:all-organisations')) {
            $query = $query->inOrganisation(Auth::user()->getOrganisationId());
        }
        $query = $query->orderBy('created_at','desc');
        return $query->get();
    }

}