<?php namespace Concore\Sam\Models;

use Auth, Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection;

class Task extends \Concore\Personnel\Models\Task {

    public $relation_methods = ['timeslots','people_present','service_user_case','next_activity','user'];

    public $removed_relation_validation_rules = ['next_activity'];
    public $added_validation_rules = [
        'next-activity.0.task.nature' => 'sometimes|required',
        'next-activity.0.task.date' => 'sometimes|required',
        'timeslots' => 'has-many-unique:timeslot.type|timeslot_types_for_case_or_advocate',
        'user_id' => 'required'
    ];
    public $replaced_validation_rules = [
        'nature_id' => 'nature_id_for_case_or_advocate'
    ];
    public $fields_with_other = ['people_present','nature_id'];

    public $dates = ['date'];

    public function __construct($attributes = array()) {
        $this->added_validation_rules['date'] = 'before:' . Carbon::now()->addDays(1)->format('Y-m-d');
        parent::__construct($attributes);
    }

    public function getNameAttribute() {
        if($this->nature) {
            return $this->nature->name;
        } else {
            return 'Activity #' . $this->id;
        }
    }

    public function service_user_case() {
        return $this->belongsTo(app('Concore\Sam\Models\ServiceUserCase'));
    }

    public function getUserRelationChoices() {
        $query = $this->user()->getRelated()->withActiveRole();
        if(!Auth::user()->can('see:all-organisations')) {
            $query = $query->inOrganisation(Auth::user()->getOrganisationId());
        }
        return $query->get();
    }

    public function getServiceUserCaseRelationChoices() {
        $query = $this->service_user_case()->getRelated();
        $query = $query->where(function($query) {
            $query = $query->open();
            if($this->id) {
                $query->orWhere('id','=',$this->service_user_case_id);
            }
        });
        if(!Auth::user()->can('see:all-organisations')) {
            $query = $query->inOrganisation(Auth::user()->getOrganisationId());
        }
        if(!Auth::user()->hasRole('coordinator') && !Auth::user()->hasRole('manager')) {
            if(Auth::user()->hasRole('advocate')) {
                $query = $query->forAdvocate(Auth::user()->employee->advocate->id);
            }
        }
        return $query->get();
    }

    public function getPeoplePresentRelationChoices() {
        if(Auth::user()->can('see:activities-update') && !Auth::user()->can('do:activities-declare-others-present')) {
            return $this->people_present()->getRelated()->where(function($query) {
                $query->where('id','=',Auth::user()->employee->person->id);
                if(Auth::user()->hasRole('advocate')) {
                    $query->orWhereHas('service_user',function($query) {
                        $query->whereHas('cases',function($query) {
                            $query->whereIn('id',Auth::user()->employee->advocate->cases()->getRelatedIds());
                        });
                    });
                }
            })->get();
        }
        return parent::getPeoplePresentRelationChoices();
    }

    public function next_activity() {
        return $this->belongsTo(app('Concore\Sam\Models\Task'),'next_activity_id');
    }

    public function previous_activity() {
        return $this->hasOne(app('Concore\Sam\Models\Task'),'next_activity_id');
    }

    public function getSlugged() {
        return 'activity';
    }

    public function getSluggedPlural() {
        return 'activities';
    }

    public function getOutcomeSummary() {
        $words = explode(' ', $this->outcome);
        $summary = array_slice($words, 0, 50);
        return implode(' ', $summary) . (count($summary) < count($words) ? '...' : '');
    }

}