<?php namespace Concore\Sam\Reports;

use Concore\Personnel\Models\Role;
use Concore\Sam\Models\Employee;
use DB, Carbon\Carbon;
use Concore\Foundation\Models\Traits\IdentifiableTrait;
use Concore\Reports\Traits\ReportableTrait;
use Concore\Reports\Table;
use Concore\Sam\Reports\Interfaces\SamReportableInterface;
use Concore\Sam\Reports\Traits\SamReportableTrait;
use Concore\Sam\Models\ActivePeriod;

abstract class EmployeesReport extends MasterRoleReport
{

    protected $active_period;
    /**
     * @var Employee
     */
    protected $employee;

    protected $start_during_end_wording = [
        [
            'Active',
            'Never active',
            'Inactive'
        ],
        [
            'Activations',
            'Reactivations',
            'Deactivations'
        ],
        [
            'Active',
            'Never active',
            'Inactive'
        ]
    ];

    public function __construct(
        Employee $employee,
        ActivePeriod $active_period,
        Role $role
    ) {
        $this->active_period = $active_period;
        $this->employee = $employee;
        $this->role = $role;
        $this->employee_role_id = $this->role->where('name','=',$this->role_name)->first(['id'])->id;
    }

    protected function getBaseUrlArgs() {
        return [
            'employees',
            'organisation_id' => $this->organisation_id,
            'date_from' => $this->from->format('Y-m'),
            'date_to' => $this->to->format('Y-m')
        ];
    }

    public function getDates($query = null, $field = null, $format = null) {
        $query = $this->getActivePeriodBaseQuery();
        return $this->getDatesTrait($query,['start','end'],'Y-m-d');
    }

    protected function getFirstDate($query, $field, $format) {
        return $this->first = Carbon::createFromFormat('Y-m-d', '2012-01-01');
    }

    protected function getLastDate($query, $field, $format) {
        return $this->last = Carbon::now();

//        $count_query = clone $query;
//        if ($count_query->select('id')->whereNull('end')->count()) {
//            $this->last = Carbon::now();
//        }
//        else {
//            $end_query = clone $query;
//            $last = $end_query->select('end')->max('end');
//            if (!$last) {
//                $last = $query->select('start')->max('start');
//            }
//            $this->last = $last ? Carbon::createFromFormat($format, $last) : $this->first;
//        }
//        return $this->last;
    }

    public function getTables()
    {
        if (!$this->tables) {
            $tables['start_of_period_table'] = $this->getStartOfPeriodTable();
            $tables['during_period_table'] = $this->getDuringPeriodTable();
            $tables['end_of_period_table'] = $this->getEndOfPeriodTable();
            $tables['gender_totals_table'] = $this->getGenderTotalsTable();
            $tables['ethnicity_totals_table'] = $this->getEthnicityTotalsTable();
            $tables['age_totals_table'] = $this->getAgeTotalsTable();
            $this->tables = $tables;
        }
        return $this->tables;
    }

    protected function getBaseQuery() {
        $query = $this->employee;
        if (!is_null($this->organisation_id)) {
            $query = $query->inOrganisation($this->organisation_id);
        }
        return $query;
    }

    protected function getActivePeriodBaseQuery() {
        $query = $this->active_period->whereHas('role',function($role) {
            $role->where('name','=',$this->role_name);
        });
        if (!is_null($this->organisation_id)) {
            $query = $query->whereHas('employee', function($query) {
                $query->inOrganisation($this->organisation_id);
            });
        } else {
            $query = $query->has('employee');
        }
        return $query;
    }

    public function getGenderTotalsTable() {
        if (is_null($this->from)) {
            if (is_null($this->first)) {
                $this->getDates();
            }
            $from = $this->first;
        }
        else {
            $from = $this->from;
        }
        if (is_null($this->to)) {
            if (is_null($this->last)) {
                $this->getDates();
            }
            $to = $this->last;
        }
        else {
            $to = $this->to;
        }
        $employee_query = $this->getBaseQuery();
        $gender_counts = $employee_query
            ->selectRaw('IFNULL(genders.id,"undefined") as id, IFNULL(genders.name,"Undefined") as gender, count(*) as quantity')
            ->withActivePeriods($this->role_name,function($active_period) use ($from, $to) {
                $active_period->where(function($q) use ($from, $to) {
                    $q->where('start','>=',$from->format('Y-m-d'))->where('start','<=',$to->format('Y-m-d'));
                });
            })
            ->join('people', 'employees.person_id', '=', 'people.id')
            ->leftJoin('genders', 'people.gender_id', '=', 'genders.id')
            ->groupBy('genders.name')
            ->get();
        $gender_counts = $gender_counts->toArray();

        $gender_ids = array_pluck($gender_counts,'id');
        foreach($gender_counts as &$value) {
            unset($value['id']);
        }
        return new Table(
            'Gender',
            [
                '',
                'Quantity'
            ],
            $gender_counts,
            null,
            function($row_index) use($gender_ids) {
                return route('sculpt.index',$this->getBaseUrlArgs() + [
                    'roles[]' => $this->employee_role_id,
                    'gender' => $gender_ids[$row_index]
                ]);
            }
        );
    }

    public function getEthnicityTotalsTable() {
        if (is_null($this->from)) {
            if (is_null($this->first)) {
                $this->getDates();
            }
            $from = $this->first;
        }
        else {
            $from = $this->from;
        }
        if (is_null($this->to)) {
            if (is_null($this->last)) {
                $this->getDates();
            }
            $to = $this->last;
        }
        else {
            $to = $this->to;
        }
        $employee_query = $this->getBaseQuery();
        $ethnicity_counts = $employee_query
            ->selectRaw('people.ethnicity as ethnicity, count(*) as quantity')
            ->withActivePeriods($this->role_name, function($active_period) use ($from, $to) {
                $active_period->where(function($q) use ($from, $to) {
                    $q->where('start','>=',$from->format('Y-m-d'))->where('start','<=',$to->format('Y-m-d'));
                });
            })
            ->join('people', 'employees.person_id', '=', 'people.id')
            ->groupBy('people.ethnicity')
            ->get();
        $ethnicity_counts = $ethnicity_counts->toArray();
        foreach($ethnicity_counts as &$ethnicity_count) {
            if (is_null($ethnicity_count['ethnicity'])) {
                $ethnicity_count['ethnicity'] = 'Undefined';
            }
        }
        return new Table(
            'Ethnicity',
            [
                '',
                'Quantity'
            ],
            $ethnicity_counts,
            null,
            function($row_index, $row) {
                return route('sculpt.index',$this->getBaseUrlArgs() + [
                    'roles[]' => $this->employee_role_id,
                    'ethnicity' => $row['ethnicity'] !== 'Undefined' ? $row['ethnicity'] : 'undefined',
                ]);
            }
        );
    }

    public function getAgeTotalsTable() {
        if (is_null($this->from)) {
            if (is_null($this->first)) {
                $this->getDates();
            }
            $from = $this->first;
        }
        else {
            $from = $this->from;
        }
        if (is_null($this->to)) {
            if (is_null($this->last)) {
                $this->getDates();
            }
            $to = $this->last;
        }
        else {
            $to = $this->to;
        }
        $employee_query = $this->getBaseQuery();
        $ages_grouped = $employee_query->selectAgeGroups()->withActivePeriods($this->role_name, function($active_period) use ($from, $to) {
            $active_period->where(function($q) use ($from, $to) {
                $q->where('start','>=',$from->format('Y-m-d'))->where('start','<=',$to->format('Y-m-d'));
            });
        })->first()->toArray();
        $age_counts = [];
        if(count(array_filter($ages_grouped))) {
            foreach ($ages_grouped as $grouping => $count) {
                $age_counts[] = [
                    'group' => $grouping,
                    'count' => $count
                ];
            }
        }
        return new Table(
            'Age',
            [
                '',
                'Quantity'
            ],
            $age_counts,
            null,
            function($row_index, $row) {
                return route('sculpt.index',$this->getBaseUrlArgs() + [
                    'roles[]' => $this->employee_role_id,
                    'age' => $row['group'] !== 'Undefined' ? $row['group'] : 'undefined',
                ]);
            }
        );
    }

}