<?php namespace Concore\Sam\Reports;

use DB, Carbon\Carbon;
use Concore\Foundation\Models\Traits\IdentifiableTrait;
use Concore\Reports\Traits\ReportableTrait;
use Concore\Reports\Table;
use Concore\Sam\Reports\Interfaces\SamReportableInterface;
use Concore\Sam\Reports\Traits\SamReportableTrait;
use Concore\Sam\Models\ServiceUser;
use Concore\Sam\Models\ServiceUserCase;
use Concore\Sam\Models\ActivePeriod;

class ServiceUserCasesReport extends MasterRoleReport
{

    private $service_user;
    private $service_user_case;
    private $active_period;

    public function __construct(
        ServiceUser $service_user,
        ServiceUserCase $service_user_case,
        ActivePeriod $active_period
    ) {
        $this->service_user = $service_user;
        $this->service_user_case = $service_user_case;
        $this->active_period = $active_period;
    }

    protected $role_name = 'service-user-case';

    protected $start_during_end_wording = [
        [
            'Open cases',
            'Cases yet to be opened',
            'Closed cases'
        ],
        [
            'First time case openings',
            'Case reopenings',
            'Case closings'
        ],
        [
            'Open cases',
            'Cases yet to be opened',
            'Closed cases'
        ]
    ];

    public function getNameAttribute() {
        return 'Service User Cases';
    }

    public function getDates($query = null, $field = null, $format = null) {
        $query = $this->active_period->has('service_user_case');
        if (!is_null($this->organisation_id)) {
            $query = $query->whereHas('service_user_case',function($query) {
                $query->inOrganisation($this->organisation_id);
            });
        }
        return $this->getDatesTrait($query,['start','end'],'Y-m-d');
    }

    protected function getFirstDate($query, $field, $format) {
        return $this->first = Carbon::createFromFormat('Y-m-d', '2012-01-01');
    }

    protected function getLastDate($query, $field, $format) {
        return $this->last = Carbon::now();

//        $count_query = clone $query;
//        if ($count_query->select('id')->whereNull('end')->count()) {
//            $this->last = Carbon::now();
//        }
//        else {
//            $end_query = clone $query;
//            $last = $end_query->select('end')->max('end');
//            if (!$last) {
//                $last = $query->select('start')->max('start');
//            }
//            $this->last = $last ? Carbon::createFromFormat($format, $last) : $this->first;
//        }
//        return $this->last;
    }

    public function getTables() {
        if (!$this->tables) {
            $tables['start_of_period_table'] = $this->getStartOfPeriodTable();
            $tables['during_period_table'] = $this->getDuringPeriodTable();
            $tables['end_of_period_table'] = $this->getEndOfPeriodTable();
            $tables['case_types_table'] = $this->getCaseTypesTable();
            $tables['activities_table'] = $this->getActivitiesTable();
            $tables['gender_totals_table'] = $this->getGenderTotalsTable();
            $tables['referral_totals_table'] = $this->getReferralTotalsTable();
            $tables['referral_methods_totals_table'] = $this->getReferralMethodsTotalsTable();
            $tables['journey_stage_totals_table'] = $this->getJourneyStageTotalsTable();
            $tables['cancer_type_totals_table'] = $this->getCancerTypeTotalsTable();
            $tables['advocacy_issue_totals_table'] = $this->getAdvocacyIssueTotalsTable();
            $tables['how_they_heard_totals_table'] = $this->getHowTheyHeardTotalsTable();
            $tables['ethnicity_totals_table'] = $this->getEthnicityTotalsTable();
            $tables['age_totals_table'] = $this->getAgeTotalsTable();
            $this->tables = $tables;
        }
        return $this->tables;
    }

    protected function getBaseQuery() {
        $query = $this->service_user_case;
        if (!is_null($this->organisation_id)) {
            $query = $query->inOrganisation($this->organisation_id);
        }
        return $query;
    }

    protected function getActivePeriodBaseQuery() {
        $query = $this->active_period;
        if (!is_null($this->organisation_id)) {
            $query = $query->whereHas('service_user_case', function($query) {
                $query->inOrganisation($this->organisation_id);
            });
        } else {
            $query = $query->has('service_user_case');
        }
        return $query;
    }

    public function getCaseTypesTable() {
        list($from, $to) = $this->setupDates();
        $query = $this->getBaseQuery();

        $counts = $query->selectRaw('case_type, COUNT(*) as count')->whereHas('active_periods', function($active_period) use ($from, $to) {
            $active_period->where(function($q) use ($from, $to) {
                $q->where('start','>=',$from->format('Y-m-d'))->where('start','<=',$to->format('Y-m-d'));
            });
        })->selectRaw('case_type')->groupBy('case_type')->get()->toArray();

        return new Table(
            'Case type',
            [
                '',
                'Quantity'
            ],
            $counts
        );
    }

    public function getActivitiesTable() {
        list($from, $to) = $this->setupDates();
        $case_ids = false;

        if (!is_null($this->organisation_id)) {
            $query = $this->getBaseQuery();
            $case_ids = $query->lists('id');
        }

        $activities_count = DB::table('tasks')
            ->select(DB::raw('count(*) as count'))
            ->join('service_user_cases', 'tasks.service_user_case_id', '=', 'service_user_cases.id')
            ->where('tasks.date','>=',$from->format('Y-m-d'))
            ->where('tasks.date','<=',$to->format('Y-m-d'))
            ->groupBy('service_user_cases.id');

        if ($case_ids) {
            $activities_count = $activities_count->whereIn('service_user_cases.id', $case_ids);
        }

        $average_activities_count = DB::table(DB::raw('('.$activities_count->toSql().') as tasks'))
            ->select(DB::raw('avg(tasks.count) as average'))
            ->mergeBindings($activities_count)
            ->first();

        $activities_time = DB::table('tasks')
            ->select(DB::raw('sum(duration) as time'))
            ->join('service_user_cases', 'tasks.service_user_case_id', '=', 'service_user_cases.id')
            ->where('tasks.date','>=',$from->format('Y-m-d'))
            ->where('tasks.date','<=',$to->format('Y-m-d'))
            ->groupBy('service_user_cases.id');

        if ($case_ids) {
            $activities_time = $activities_time->whereIn('service_user_cases.id', $case_ids);
        }

        $average_activities_time = DB::table(DB::raw('('.$activities_time->toSql().') as tasks'))
            ->select(DB::raw('avg(tasks.time) as average'))
            ->mergeBindings($activities_time)
            ->first();

        return new Table(
            'Activities',
            [],
            [
                [
                    'Average number of activities received during this period*',
                    round($average_activities_count->average, 1)
                ],
                [
                    "Average total hours' support received during this period*",
                    (int) $average_activities_time->average
                ]
            ],
            [
                1 => [
                    1 => $this->formatDuration()
                ]
            ]
        );
    }

    public function getGenderTotalsTable() {
        list($from, $to) = $this->setupDates();
        $query = $this->getBaseQuery();
        $gender_counts = $query->selectRaw('IFNULL(genders.id,"undefined") as id, IFNULL(genders.name,"Undefined") as gender, count(*) as quantity')->whereHas('active_periods', function($active_period) use ($from, $to) {
            $active_period->where(function($q) use ($from, $to) {
                $q->where('start','>=',$from->format('Y-m-d'))->where('start','<=',$to->format('Y-m-d'));
            });
        })
            ->join('service_users', 'service_user_cases.service_user_id', '=', 'service_users.id')
            ->join('people', 'service_users.person_id', '=', 'people.id')
            ->leftJoin('genders', 'people.gender_id', '=', 'genders.id')
            ->groupBy('genders.name')
            ->get();
        $gender_counts = $gender_counts->toArray();

        $gender_ids = array_pluck($gender_counts,'id');
        foreach($gender_counts as &$value) {
            unset($value['id']);
        }

        return new Table(
            'Gender',
            [
                '',
                'Quantity'
            ],
            $gender_counts,
            null,
            function($row_index) use($gender_ids) {
                return route('sculpt.index',[
                    'service-user-cases',
                    'organisation_id' => $this->organisation_id,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'gender' => $gender_ids[$row_index]
                ]);
            }
        );
    }

    public function getReferralTotalsTable() {
        list($from, $to) = $this->setupDates();
        $query = $this->getBaseQuery();
        $referral_counts = $query->selectRaw('
            IFNULL(referral_sources.id,
                (CASE WHEN service_user_cases.referral_source_other IS NOT NULL THEN "other" ELSE "undefined" END)
            ) as id,
            IFNULL(referral_sources.name,
                (CASE WHEN service_user_cases.referral_source_other IS NOT NULL THEN "Other" ELSE "Undefined" END)
            ) as referral_source,
            count(*) as quantity
        ')->whereHas('active_periods', function($active_period) use ($from, $to) {
            $active_period->where(function($q) use ($from, $to) {
                $q->where('start','>=',$from->format('Y-m-d'))->where('start','<=',$to->format('Y-m-d'));
            });
        })
            ->leftJoin('referral_sources', 'service_user_cases.referral_source_id', '=', 'referral_sources.id')
            ->groupBy('referral_source')
            ->get()->toArray();
        $referral_ids = array_pluck($referral_counts,'id');
        foreach($referral_counts as &$value) {
            unset($value['id']);
        }

        return new Table(
            'Referral source',
            [
                '',
                'Quantity'
            ],
            $referral_counts,
            null,
            function($row_index) use($referral_ids) {
                return route('sculpt.index',[
                    'service-user-cases',
                    'organisation_id' => $this->organisation_id,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'referral_source' => $referral_ids[$row_index]
                ]);
            }
        );
    }

    public function getReferralMethodsTotalsTable() {
        list($from, $to) = $this->setupDates();
        $query = $this->getBaseQuery();
        $referral_counts = $query->selectRaw('
            IFNULL(referral_methods.id,
                (CASE WHEN service_user_cases.referral_method_other IS NOT NULL THEN "other" ELSE "undefined" END)
            ) as id,
            IFNULL(referral_methods.name,
                (CASE WHEN service_user_cases.referral_method_other IS NOT NULL THEN "Other" ELSE "Undefined" END)
            ) as referral_method,
            count(*) as quantity
        ')->whereHas('active_periods', function($active_period) use ($from, $to) {
            $active_period->where(function($q) use ($from, $to) {
                $q->where('start','>=',$from->format('Y-m-d'))->where('start','<=',$to->format('Y-m-d'));
            });
        })
            ->leftJoin('referral_methods', 'service_user_cases.referral_method_id', '=', 'referral_methods.id')
            ->groupBy('referral_method')
            ->get()->toArray();
        $referral_ids = array_pluck($referral_counts,'id');
        foreach($referral_counts as &$value) {
            unset($value['id']);
        }

        return new Table(
            'Referral method',
            [
                '',
                'Quantity'
            ],
            $referral_counts,
            null,
            function($row_index) use($referral_ids) {
                return route('sculpt.index',[
                    'service-user-cases',
                    'organisation_id' => $this->organisation_id,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'referral_method' => $referral_ids[$row_index]
                ]);
            }
        );
    }

    public function getHowTheyHeardTotalsTable() {
        list($from, $to) = $this->setupDates();
        $query = $this->getBaseQuery();

        $how_they_heard_counts = $query->selectRaw('
            IFNULL(how_they_heard,
                (CASE WHEN how_they_heard_other IS NOT NULL THEN "Other" ELSE "Undefined" END)
            ) as _how_they_heard,
            count(*) as quantity
        ')->whereHas('active_periods', function($active_period) use ($from, $to) {
            $active_period->where(function($q) use ($from, $to) {
                $q->where('start','>=',$from->format('Y-m-d'))->where('start','<=',$to->format('Y-m-d'));
            });
        })
            ->groupBy('_how_they_heard')
            ->get()->toArray();

        return new Table(
            'How the service user heard about the project',
            [
                '',
                'Quantity'
            ],
            $how_they_heard_counts,
            null,
            function($row_index, $row) {
                return route('sculpt.index',[
                    'service-user-cases',
                    'organisation_id' => $this->organisation_id,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'how_they_heard' => $row['_how_they_heard'] !== 'Undefined' ? $row['_how_they_heard'] : 'Undefined',
                ]);
            }
        );
    }

    public function getJourneyStageTotalsTable() {
        list($from, $to) = $this->setupDates();
        $query = $this->getBaseQuery();
        $journey_stage_counts = $query->selectRaw('
            IFNULL(journey_stages.id,
                (CASE WHEN service_user_cases.journey_stage_other IS NOT NULL THEN "Other" ELSE "Undefined" END)
            ) as id,
            IFNULL(journey_stages.name,
                (CASE WHEN service_user_cases.journey_stage_other IS NOT NULL THEN "Other" ELSE "Undefined" END)
            ) as journey_stage,
            count(*) as quantity
        ')->whereHas('active_periods', function($active_period) use ($from, $to) {
            $active_period->where(function($q) use ($from, $to) {
                $q->where('start','>=',$from->format('Y-m-d'))->where('start','<=',$to->format('Y-m-d'));
            });
        })
            ->leftJoin('journey_stages', 'service_user_cases.journey_stage_Id', '=', 'journey_stages.id')
            ->groupBy('journey_stage')
            ->get()->toArray();
        $journey_stage_ids = array_pluck($journey_stage_counts,'id');
        foreach($journey_stage_counts as &$value) {
            unset($value['id']);
        }

        return new Table(
            'Journey Stage',
            [
                '',
                'Quantity'
            ],
            $journey_stage_counts,
            null,
            function($row_index) use($journey_stage_ids) {
                return route('sculpt.index',[
                    'service-user-cases',
                    'organisation_id' => $this->organisation_id,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'journey_stage' => $journey_stage_ids[$row_index]
                ]);
            }
        );
    }

    public function getCancerTypeTotalsTable() {
        list($from, $to) = $this->setupDates();
        // Subquery to only use cases with active periods started within date range, used in both queries below
        $active_periods = DB::table('active_periods')
            ->selectRaw('count(*)')
            ->join('active_periodables', 'active_periods.id', '=', 'active_periodables.active_period_id')
            ->where('active_periods.start','>=',$from->format('Y-m-d'))->where('active_periods.start','<=',$to->format('Y-m-d'))
            ->where('active_periodables.active_periodables_type','=','Concore\Sam\Models\ServiceUserCase')
            ->whereRaw('`active_periodables`.`active_periodables_id` = `service_user_cases`.`id`');
        // Query 1: Cancer types and Undefined (ignoring 'other')
        $query_cancer_types_and_undefined = DB::table('service_user_cases')->selectRaw('
            IFNULL(cancer_types.id,"Undefined") as cancer_type_id,
            IFNULL(cancer_types.name,"Undefined") as cancer_type,
            count(*) as quantity
        ')
            ->leftJoin('cancer_type_service_user_case', 'service_user_cases.id', '=', 'cancer_type_service_user_case.service_user_case_id')
            ->leftJoin('cancer_types', 'cancer_type_service_user_case.cancer_type_id', '=', 'cancer_types.id')
            ->whereRaw('(' . $active_periods->toSql() . ') >= 1')
            ->mergeBindings($active_periods)
            ->where(function($query) {
                $query->whereNull('cancer_types_other')->orWhereNotNull('cancer_types.name');
            })
            ->groupBy('cancer_type');
        // Query 2: Count up 'other'. Need to do this separately as some of the above will have 'other', but we want to count the
        // ones with other in their own right
        $query_other = DB::table('service_user_cases')->selectRaw('
            "Other" as cancer_type_id,
            "Other" as cancer_type,
            count(*) as quantity
        ')
            ->whereRaw('(' . $active_periods->toSql() . ') >= 1')
            ->mergeBindings($active_periods)
            ->whereNotNull('cancer_types_other');
        // Apply organisation constraints
        if(!is_null($this->organisation_id)) {
            $people = DB::table('people')
                ->selectRaw('count(*)')
                ->whereRaw('`service_users`.`person_id` = `people`.`id`')
                ->where('people.organisation_id', '=', $this->organisation_id);
            $organisation = DB::table('service_users')
                ->selectRaw('count(*)')
                ->whereRaw('`service_user_cases`.`service_user_id` = `service_users`.`id`')
                ->whereRaw('(' . $people->toSql() . ') >= 1')
                ->mergeBindings($people);
            $query_cancer_types_and_undefined = $query_cancer_types_and_undefined
                ->whereRaw('(' . $organisation->toSql() . ') >= 1')
                ->mergeBindings($organisation);
            $query_other = $query_other
                ->whereRaw('(' . $organisation->toSql() . ') >= 1')
                ->mergeBindings($organisation);
        }
        // Union
        $cancer_type_counts_raw = $query_cancer_types_and_undefined->union($query_other)->get();
        $cancer_type_counts = [];
        foreach($cancer_type_counts_raw as $row) {
            $cancer_type_ids[] = $row->cancer_type_id;
            $cancer_type_counts[] = [
                'cancer_type' => $row->cancer_type,
                'quantity' => $row->quantity
            ];
        }

        return new Table(
            'Cancer Types',
            [
                '',
                'Quantity'
            ],
            $cancer_type_counts,
            null,
            function($row_index) use($cancer_type_ids) {
                return route('sculpt.index',[
                    'service-user-cases',
                    'organisation_id' => $this->organisation_id,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'cancer_type' => $cancer_type_ids[$row_index]
                ]);
            }
        );
    }

    public function getAdvocacyIssueTotalsTable() {
        // This is almost identical to the function above, do a diff to see
        list($from, $to) = $this->setupDates();
        // Subquery to only use cases with active periods started within date range, used in both queries below
        $active_periods = DB::table('active_periods')
            ->selectRaw('count(*)')
            ->join('active_periodables', 'active_periods.id', '=', 'active_periodables.active_period_id')
            ->where('active_periods.start','>=',$from->format('Y-m-d'))->where('active_periods.start','<=',$to->format('Y-m-d'))
            ->where('active_periodables.active_periodables_type','=','Concore\Sam\Models\ServiceUserCase')
            ->whereRaw('`active_periodables`.`active_periodables_id` = `service_user_cases`.`id`');
        // Query 1: Advocacy issues and Undefined (ignoring 'other')
        $query_advocacy_issues_and_undefined = DB::table('service_user_cases')->selectRaw('
            IFNULL(advocacy_issues.id,"Undefined") as advocacy_issue_id,
            IFNULL(advocacy_issues.name,"Undefined") as advocacy_issue,
            count(*) as quantity
        ')
            ->leftJoin('advocacy_issue_service_user_case', 'service_user_cases.id', '=', 'advocacy_issue_service_user_case.service_user_case_id')
            ->leftJoin('advocacy_issues', 'advocacy_issue_service_user_case.advocacy_issue_id', '=', 'advocacy_issues.id')
            ->whereRaw('(' . $active_periods->toSql() . ') >= 1')
            ->mergeBindings($active_periods)
            ->where(function($query) {
                $query->whereNull('advocacy_issues_other')->orWhereNotNull('advocacy_issues.name');
            })
            ->groupBy('advocacy_issue');
        // Query 2: Count up 'other'. Need to do this separately as some of the above will have 'other', but we want to count the
        // ones with other in their own right
        $query_other = DB::table('service_user_cases')->selectRaw('
            "Other" as advocacy_issue_id,
            "Other" as advocacy_issue,
            count(*) as quantity
        ')
            ->whereRaw('(' . $active_periods->toSql() . ') >= 1')
            ->mergeBindings($active_periods)
            ->whereNotNull('advocacy_issues_other');
        // Apply organisation constraints
        if(!is_null($this->organisation_id)) {
            $people = DB::table('people')
                ->selectRaw('count(*)')
                ->whereRaw('`service_users`.`person_id` = `people`.`id`')
                ->where('people.organisation_id', '=', $this->organisation_id);
            $organisation = DB::table('service_users')
                ->selectRaw('count(*)')
                ->whereRaw('`service_user_cases`.`service_user_id` = `service_users`.`id`')
                ->whereRaw('(' . $people->toSql() . ') >= 1')
                ->mergeBindings($people);
            $query_advocacy_issues_and_undefined = $query_advocacy_issues_and_undefined
                ->whereRaw('(' . $organisation->toSql() . ') >= 1')
                ->mergeBindings($organisation);
            $query_other = $query_other
                ->whereRaw('(' . $organisation->toSql() . ') >= 1')
                ->mergeBindings($organisation);
        }
        // Union
        $advocacy_issue_counts_raw = $query_advocacy_issues_and_undefined->union($query_other)->get();
        $advocacy_issue_counts = [];
        foreach($advocacy_issue_counts_raw as $row) {
            $advocacy_issue_ids[] = $row->advocacy_issue_id;
            $advocacy_issue_counts[] = [
                'advocacy_issue' => $row->advocacy_issue,
                'quantity' => $row->quantity
            ];
        }

        return new Table(
            'Advocacy Issues',
            [
                '',
                'Quantity'
            ],
            $advocacy_issue_counts,
            null,
            function($row_index) use($advocacy_issue_ids) {
                return route('sculpt.index',[
                    'service-user-cases',
                    'organisation_id' => $this->organisation_id,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'advocacy_issue' => $advocacy_issue_ids[$row_index]
                ]);
            }
        );
    }

    public function getEthnicityTotalsTable() {
        list($from, $to) = $this->setupDates();
        $query = $this->getBaseQuery();
        $ethnicity_counts = $query->selectRaw('IFNULL(people.ethnicity,\'Undefined\') as ethnicity, count(*) as quantity')->whereHas('active_periods', function($active_period) use ($from, $to) {
            $active_period->where(function($q) use ($from, $to) {
                $q->where('start','>=',$from->format('Y-m-d'))->where('start','<=',$to->format('Y-m-d'));
            });
        })
            ->join('service_users', 'service_user_cases.service_user_id', '=', 'service_users.id')
            ->join('people', 'service_users.person_id', '=', 'people.id')
            ->groupBy('people.ethnicity');
        $ethnicity_counts = $ethnicity_counts->get();
        $ethnicity_counts = $ethnicity_counts->toArray();
        return new Table(
            'Ethnicity',
            [
                '',
                'Quantity'
            ],
            $ethnicity_counts,
            null,
            function($row_index, $row) {
                return route('sculpt.index',[
                    'service-user-cases',
                    'organisation_id' => $this->organisation_id,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'ethnicity' => $row['ethnicity'] !== 'Undefined' ? $row['ethnicity'] : 'undefined',
                ]);
            }
        );
    }

    public function getAgeTotalsTable() {
        list($from, $to) = $this->setupDates();
        $query = $this->getBaseQuery();
        $ages_grouped = $query->selectAgeGroups()->whereHas('active_periods', function($active_period) use ($from, $to) {
            $active_period->where(function($q) use ($from, $to) {
                $q->where('start','>=',$from->format('Y-m-d'))->where('start','<=',$to->format('Y-m-d'));
            });
        })->first()->toArray();
        $age_counts = [];
        if(count(array_filter($ages_grouped))) {
            foreach($ages_grouped as $grouping => $count) {
                $age_counts[] = [
                    'group' => $grouping,
                    'count' => $count
                ];
            }
        }
        return new Table(
            'Age',
            [
                '',
                'Quantity'
            ],
            $age_counts,
            null,
            function($row_index, $row) {
                return route('sculpt.index',[
                    'service-user-cases',
                    'organisation_id' => $this->organisation_id,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'age' => $row['group'] !== 'Undefined' ? $row['group'] : 'undefined',
                ]);
            }
        );
    }

    public function setupDates() {
        if (is_null($this->from)) {
            if (is_null($this->first)) {
                $this->getDates();
            }
            $from = $this->first;
        }
        else {
            $from = $this->from;
        }
        if (is_null($this->to)) {
            if (is_null($this->last)) {
                $this->getDates();
            }
            $to = $this->last;
        }
        else {
            $to = $this->to;
        }
        return [$from, $to];
    }

}