<?php namespace Concore\Sam\Reports;

use Concore\Personnel\Models\Role;
use Concore\Sam\Models\Employee;
use DB, Carbon\Carbon;
use Concore\Foundation\Models\Traits\IdentifiableTrait;
use Concore\Reports\Traits\ReportableTrait;
use Concore\Reports\Table;
use Concore\Sam\Reports\Interfaces\SamReportableInterface;
use Concore\Sam\Reports\Traits\SamReportableTrait;
use Concore\Sam\Models\ActivePeriod;

abstract class MasterRoleReport implements SamReportableInterface
{

    use IdentifiableTrait;
    use ReportableTrait {
        getDates as getDatesTrait;
        ReportableTrait::getUrl insteadof IdentifiableTrait;
    }
    use SamReportableTrait;

    public function __construct() {
        $this->setupDates();
    }

    // Returns query looking for active periods that match our chosen time period
    // To be used as a subquery with >= 1 to find matching active_periodables (e.g. cases, employees)

    public function getBaseQueryBuilder() {

        $query = $this->getTable();
        
        $query = $this->applyOrganisationConstraints($query);

        $query = $this->applyActivePeriodConstraints($query);

        return $query;
        
    }

    public function getTable() {
        return DB::table($this->table);
    }

    public function getActivePeriodsSubQuery() {
        return DB::table('active_periods')
            ->selectRaw('count(*)')
            ->join('active_periodables', 'active_periods.id', '=', 'active_periodables.active_period_id')
            ->where('active_periodables.active_periodables_type','=',get_class($this->model))
            ->whereRaw("`active_periodables`.`active_periodables_id` = `{$this->table}`.`id`");
    }
    
    public function getActivePeriodQueryBuilder($where = null) {

        $active_periods = $this->getActivePeriodsSubQuery();

        // If no where function specified, find all with active periods that started inside the time period
        if(!$where) {
            $where = function($query) {
                $query->where('active_periods.start','>=',$this->from->format('Y-m-d'))
                    ->where('active_periods.start','<=',$this->to->format('Y-m-d'));
                return $query;
            };
        }

        $active_periods = $where($active_periods);

        return $active_periods;

    }

    public function applyActivePeriodConstraints($query, $where = null, $operator = ">= 1") {
        $active_periods_query_builder = $this->getActivePeriodQueryBuilder($where);
        return $query->whereRaw('(' . $active_periods_query_builder->toSql() . ') ' . $operator)
            ->mergeBindings($active_periods_query_builder);
    }

    public function applyNegativeActivePeriodConstraints($query, $where = null) {
        return $this->applyActivePeriodConstraints($query, $where, "< 1");
    }

    protected function applyOrganisationConstraints($query) {
        $query = $this->joinToPeopleTable($query);
        if($this->organisation_id) {
            $query = $query->where('people.organisation_id', '=', $this->organisation_id);
        }
        return $query;
    }

    // People tables (shared by employees and cases)

    public function getAgeTable() {

        $query_string = "SUM(CASE WHEN people.age IS NULL OR people.age = 0 THEN 1 ELSE 0 END) as 'Undefined'";
        foreach(config('sam.reporting-age-groups') as $age_group) {
            list($from,$to) = explode('-',$age_group);
            $query_string .= ", SUM(IF(people.age BETWEEN $from AND $to,1,0)) as '$age_group'";
        }

        $query = $this->getBaseQueryBuilder();

        $ages_grouped = $query
            ->selectRaw($query_string)
            ->first();
        $ages_grouped = get_object_vars($ages_grouped);

        $age_counts = [];
        if(count(array_filter($ages_grouped))) {
            foreach($ages_grouped as $grouping => $count) {
                $age_counts[] = [
                    'group' => $grouping,
                    'count' => $count
                ];
            }
        }

        return new Table(
            'Age',
            [
                '',
                'Quantity'
            ],
            $age_counts
        );

    }

    public function getGenderTable() {

        $query = $this->getBaseQueryBuilder();

        $gender_counts = $query
            ->selectRaw('IFNULL(genders.id,"undefined") as id, IFNULL(genders.name,"Undefined") as gender, count(*) as quantity')
            ->leftJoin('genders', 'people.gender_id', '=', 'genders.id')
            ->groupBy('genders.name')
            ->get();

        $ids = array_pluck($gender_counts,'id');
        foreach($gender_counts as &$value) {
            unset($value->id);
        }

        $this->ids = $ids;

        return new Table(
            'Gender',
            [
                '',
                'Quantity'
            ],
            $gender_counts
        );

    }

    public function getEthnicityTable() {

        return new Table(
            'Ethnicity',
            [
                '',
                'Quantity'
            ],
            $this->getTotalForEnumOnPerson('ethnicity')
        );

    }

    public function getSexualityTable() {

        list($counts, $ids) = $this->getTotalForRelationOnPerson('sexuality', 'sexualities', 'sexuality_id');

        $this->ids = $ids;

        return new Table(
            'Sexuality',
            [
                '',
                'Quantity'
            ],
            $counts
        );
    }

    // Helper functions

    protected function getTotalForRelationOnPerson($relation_name, $relation_table, $foreign_key) {

        $query = $this->getBaseQueryBuilder();

        $counts = $query
            ->selectRaw("IFNULL($relation_table.id,'undefined') as id, IFNULL($relation_table.name,'Undefined') as $relation_name, count(*) as quantity")
            ->leftJoin($relation_table, "people.$foreign_key", '=', "$relation_table.id")
            ->groupBy("$relation_table.name")
            ->get();

        $ids = array_pluck($counts,'id');
        foreach($counts as &$value) {
            unset($value->id);
        }

        return [$counts, $ids];

    }

    protected function getTotalForEnumOnPerson($field) {

        $query = $this->getBaseQueryBuilder();

        $counts = $query
            ->selectRaw("
                IFNULL(people.$field,'Undefined') as _$field,
                count(*) as quantity
            ")
            ->groupBy("people.$field")
            ->get();

        return $counts;

    }

    // Dates

    public function getDates($query = null, $field = null, $format = null) {
        return $this->getDatesTrait($query,['start','end'],'Y-m-d');
    }

    protected function getFirstDate() {
        return $this->first = Carbon::createFromFormat('Y-m-d', '2012-01-01');
    }

    protected function getLastDate() {
        return $this->last = Carbon::now();
    }

    public function setupDates() {
        if (is_null($this->from)) {
            if (is_null($this->first)) {
                $this->getDates();
            }
            $this->from = $this->first;
        }
        if (is_null($this->to)) {
            if (is_null($this->last)) {
                $this->getDates();
            }
            $this->to = $this->last;
        }
    }

    // Progression tables
    // TODO: Update to new format

    /*

    public function getStartOfPeriodTable() {
        list($from, $to) = $this->setupDates();
        $query = $this->getBaseQuery();
        $active_at_start = $query->withActivePeriods($this->role_name, function($active_period) use ($from, $to) {
            return $active_period
                ->where('start','<',$from->format('Y-m-d'))
                ->where(function($q) use ($from) {
                    $q->whereNull('end')
                        ->orWhere('end','>=',$from->format('Y-m-d'));
                });
        })->count();

        $query = $this->getBaseQuery();
        $never_activated_at_start = $query->where('created_at','<',$from->format('Y-m-d'))->withActivePeriods($this->role_name, function($active_period) use ($from) {
            return $active_period->where('start','<',$from->format('Y-m-d'));
        },'<',1)->count();

        $query = $this->getBaseQuery();
        $inactive_at_start = $query->withActivePeriods($this->role_name, function($active_period) use ($from) {
            return $active_period->where('end','<',$from->format('Y-m-d'));
        })->withActivePeriods($this->role_name, function($active_period) use ($from) {
            return $active_period
                ->where('start','<',$from->format('Y-m-d'))
                ->where(function($q) use ($from) {
                    $q->whereNull('end')
                        ->orWhere('end','>=',$from->format('Y-m-d'));
                });
        },'<',1)->count();

        return new Table(
            'Start of period',
            [],
            [
                [
                    $this->start_during_end_wording[0][0],
                    $active_at_start
                ],
                [
                    $this->start_during_end_wording[0][1],
                    $never_activated_at_start
                ],
                [
                    $this->start_during_end_wording[0][2],
                    $inactive_at_start
                ]
            ]
        );
    }

    public function getDuringPeriodTable() {
        list($from, $to) = $this->setupDates();
        // Get active periods that are the first for their case and are opened within the period
        $query = $this->getActivePeriodBaseQuery();
        $case_openings = $query->where('start', '>=', $from->format('Y-m-d'))->where('start', '<=', $to->format('Y-m-d'))->where('is_first',true)->count();
        // Get active periods that are not the first for their case, that are opened within the period
        $query = $this->getActivePeriodBaseQuery();
        $case_reopenings = $query->where('start', '>=', $from->format('Y-m-d'))->where('start', '<=', $to->format('Y-m-d'))->where('is_first',false)->count();
        // Get cases added but not opened yet
        $query = $this->getBaseQuery();
        $added_but_not_opened = $query->where('created_at','>=',$from->format('Y-m-d'))->where('created_at','<=',$to->format('Y-m-d'))->count();
        // Get active periods that end within the period
        $query = $this->getActivePeriodBaseQuery();
        $case_closings = $query->where('end', '>=', $from->format('Y-m-d'))->where('end', '<=', $to->format('Y-m-d'))->count();
        return new Table(
            'During period',
            [],
            [
                [
                    $this->start_during_end_wording[1][0],
                    $case_openings
                ],
                [
                    $this->start_during_end_wording[1][1],
                    $case_reopenings
                ],
                [
                    $this->start_during_end_wording[1][2],
                    $added_but_not_opened
                ],
                [
                    $this->start_during_end_wording[1][3],
                    $case_closings
                ]
            ]
        );
    }

    public function getEndOfPeriodTable() {
        if (is_null($this->from)) {
            if (is_null($this->first)) {
                $this->getDates();
            }
            $from = $this->first;
        }
        else {
            $from = $this->from;
        }
        if (is_null($this->to)) {
            if (is_null($this->last)) {
                $this->getDates();
            }
            $to = $this->last;
        }
        else {
            $to = $this->to;
        }

        $query = $this->getBaseQuery();
        $active_at_end = $query->withActivePeriods($this->role_name, function($active_period) use ($to) {
            return $active_period
                ->where('start','<=',$to->format('Y-m-d'))
                ->where(function($q) use ($to) {
                    $q->whereNull('end')
                        ->orWhere('end','>',$to->format('Y-m-d'));
                });
        })->count();

        $query = $this->getBaseQuery();
        $never_active_at_end = $query->withActivePeriods($this->role_name, function($active_period) use ($to) {
            return $active_period->where('start','<=',$to->format('Y-m-d'));
        },'<',1)->count();

        $query = $this->getBaseQuery();
        $inactive_at_end = $query->withActivePeriods($this->role_name, function($active_period) use ($to) {
            return $active_period
                ->where('end','<=',$to->format('Y-m-d'));
        })->withActivePeriods($this->role_name, function($active_period) use ($to) {
            return $active_period
                ->where('start','<=',$to->format('Y-m-d'))
                ->where(function($q) use ($to) {
                    $q->whereNull('end')
                        ->orWhere('end','>',$to->format('Y-m-d'));
                });
        },'<',1)->count();

        return new Table(
            'End of period',
            [],
            [
                [
                    $this->start_during_end_wording[2][0],
                    $active_at_end
                ],
                [
                    $this->start_during_end_wording[2][1],
                    $never_active_at_end
                ],
                [
                    $this->start_during_end_wording[2][2],
                    $inactive_at_end
                ]
            ]
        );
    }

    */

}