<?php namespace Concore\Sam\Models;

use Auth, Carbon\Carbon, DB;
use Concore\Foundation\Models\Interfaces\IdentifiableInterface;
use Concore\Foundation\Models\Traits\IdentifiableTrait;
use Concore\Notes\Models\Traits\HasNotesTrait;
use Concore\Personnel\Models\Traits\HasActivePeriodsTrait;
use Illuminate\Database\Eloquent\Model;

class ServiceUserCase extends Model implements IdentifiableInterface {

    use IdentifiableTrait,
        HasNotesTrait,
        HasActivePeriodsTrait;

	protected $guarded = ['id','created_at','updated_at'];

    public $dates = ['created_at','updated_at','last_activity'];
    public $relation_methods = ['cancer_types','advocacy_issues','service_user','advocates','notes','active_periods','contact_entities','accommodation_type','referral_source','referral_method','journey_stage'];
    public $fields_with_other = ['accommodation_type_id','referral_source_id','referral_method_id','advocacy_issues','cancer_types','journey_stage_id','how_they_heard'];
    public $added_validation_rules = ['accommodation_type_other' => 'required_if:{accommodation_type},Other'];

    public $skip_fields_when_starting_active_period = true;

    public static function boot() {
        static::saving(function($record) {
            // Only update last_activity if it's later than the current one
            $current = $record->find($record->id);
            if($record->last_activity && $current->last_activity && $current->last_activity > $record->last_activity) {
                $record->last_activity = $current->last_activity;
            }
        });
    }

    public function service_user() {
        return $this->belongsTo(app('Concore\Sam\Models\ServiceUser'));
    }

    public function getServiceUserRelationChoices() {
        if(Auth::user()->can('see:all-organisations')) {
            return $this->service_user()->getRelated()->all();
        } else {
            return $this->service_user()->getRelated()->inOrganisation(Auth::user()->getOrganisationId())->get();
        }
    }

    public function advocates() {
        return $this->belongsToMany(app('Concore\Sam\Models\Advocate'));
    }

    public function getAdvocatesRelationChoices() {
        $choices = $this->advocates()->getRelated()->active();
        if(!Auth::user()->can('see:all-organisations')) {
            $choices = $choices->inOrganisation(Auth::user()->getOrganisationId());
        }
        return $choices->get();
    }

    public function getAddressAttribute() {
        if ($this->service_user->person->address) {
            return $this->service_user->person->address;
        }
        return null;
    }

    public function cancer_types() {
        return $this->belongsToMany(app('Concore\Sam\Models\CancerType'));
    }

    public function advocacy_issues() {
        return $this->belongsToMany(app('Concore\Sam\Models\AdvocacyIssue'));
    }

    public function advocacy_outcomes() {
        return $this->belongsToMany(app('Concore\Sam\Models\AdvocacyOutcome'))->withPivot('level');
    }

    public function activities() {
        return $this->hasMany(app('Concore\Sam\Models\Task'))->orderBy('date','desc');
    }

    public function contact_entities() {
        return $this->morphToMany(app('Concore\Personnel\Models\ContactEntity'),'contact_entityables')->withPivot('contact_entity_type_id','notes');
    }

    public function hasReferrerContactInfo() {
        if($this->contact_entities) {
            return count(array_filter(array_pluck($this->contact_entities->toArray(), 'pivot.contact_entity_type_id'), function ($item) {
                return $item == 4;
            })) > 0;
        } else {
            return false;
        }
    }

    public function getNameAttribute() {
        return '#' . $this->id . ' ' . $this->service_user->name;
    }

    public function scopeInOrganisation($query, $organisation_id) {
        return $query->whereHas('service_user',function($query) use($organisation_id) {
            $query->whereHas('person',function($query) use($organisation_id) {
                $query->where('organisation_id','=',$organisation_id);
            });
        });
    }

    public function scopeForAdvocate($query, $advocate_id) {
        return $query->whereHas('advocates',function($query) use($advocate_id) {
            $query->where('advocates.id','=',$advocate_id);
        });
    }

    public function scopeOpenOrYetToBeOpened($query) {
        return $query->where(function($query) {
            $query->has('active_periods','<',1)->orWhereHas('active_periods',function($query) {
                $query->whereNull('end');
            });
        });
    }

    public function scopeOpen($query) {
        return $query->whereHas('active_periods',function($query) {
            $query->whereNull('end');
        });
    }

    public function scopeClosed($query) {
        return $query->has('active_periods')->whereHas('active_periods',function($query) {
            $query->whereNull('end');
        },'<',1);
    }

    public function scopeNeverOpened($query) {
        return $query->has('active_periods','<',1);
    }

    public function scopeWithNoAdvocate($query) {
        return $query->has('advocates','<',1)->where('no_advocate_needed','=',false);
    }

    public function getActivitiesDurationAttribute() {
        return $this->activities->map(function($activity) {
            return $activity->duration;
        })->sum();
    }

    public function getActivitiesAverageDurationAttribute() {
        return $this->activities->count() > 0 ? $this->activities_duration / $this->activities->count() : 0;
    }

    public function accommodation_type() {
        return $this->belongsTo(app('Concore\Sam\Models\AccommodationType'));
    }

    public function referral_source() {
        return $this->belongsTo(app('Concore\Sam\Models\ReferralSource'));
    }

    public function referral_method() {
        return $this->belongsTo(app('Concore\Sam\Models\ReferralMethod'));
    }

    public function journey_stage() {
        return $this->belongsTo(app('Concore\Sam\Models\JourneyStage'));
    }

    public function daysSinceLastActivity() {
        return $this->last_activity ? $this->last_activity->diffInDays(Carbon::now()) : null;
    }

    public function getHowTheyHeards() {
        $type = DB::select( DB::raw("SHOW COLUMNS FROM service_user_cases WHERE Field = 'how_they_heard'") )[0]->Type;
        preg_match('/^enum\((.*)\)$/', $type, $matches);
        return array_map(function($value) {
            return trim($value,"'");
        },explode("','", $matches[1]));
    }

    public function scopeWithActivePeriods($query, $role_name, $callback = null, $operator = '>=', $num = 1) {
        return $query->whereHas('active_periods', function($active_period) use($callback, $role_name, $operator, $num) {
            $active_period
                ->whereHas('role', function ($role) use($role_name) {
                    $role->where('name', '=', $role_name);
                });
            if($callback) {
                $active_period = $callback($active_period);
            }
        },$operator,$num);
    }

    public function scopeOnlySimple($query) {
        return $query->has('advocacy_issues','<',3);
    }

    public function scopeOnlyComplex($query) {
        return $query->has('advocacy_issues','>=',3);
    }
}