<?php namespace Concore\Sam\Reports;

use DB, Carbon\Carbon;
use Concore\Foundation\Models\Traits\IdentifiableTrait;
use Concore\Reports\Traits\ReportableTrait;
use Concore\Reports\Table;
use Concore\Sam\Reports\Interfaces\SamReportableInterface;
use Concore\Sam\Reports\Traits\SamReportableTrait;
use Concore\Sam\Models\ServiceUser;
use Concore\Sam\Models\ServiceUserCase;
use Concore\Sam\Models\ActivePeriod;

class ServiceUserCasesReport extends MasterRoleReport
{

    private $active_period;
    protected $include_cases = null;

    protected $role_name = 'service-user-case';
    protected $table = 'service_user_cases';

    public function __construct(
        ServiceUserCase $service_user_case,
        ActivePeriod $active_period
    ) {
        $this->model = $service_user_case;
        $this->active_period = $active_period;
    }

    public function getNameAttribute() {
        return 'Service User Cases';
    }

    public function getTables() {
        if (!$this->tables) {

            foreach(['all', 'simple', 'complex'] as $include_cases) {
                $this->include_cases = $include_cases;
                $tables['totals_' . $include_cases] = $this->getTotals();
                $tables['case_types_' . $include_cases] = $this->getCaseTypesTable();
                $tables['activities_' . $include_cases] = $this->getActivitiesTable();

                $tables['referral_' . $include_cases] = $this->getReferralTable();
                $tables['how_they_heard_' . $include_cases] = $this->getHowTheyHeardTable();
                $tables['referral_method_' . $include_cases] = $this->getReferralMethodTable();
                // Only show this table if we're looking at 1 month, due to server load being too great for more than this
                if($this->from->diffInMonths($this->to) == 0) {
                    $tables['referral_breakdown_' . $include_cases] = $this->getReferralBreakdownTable();
                }

                $tables['journey_stage_' . $include_cases] = $this->getJourneyStageTable();
                $tables['advocacy_issues_' . $include_cases] = $this->getAdvocacyIssuesTable();
                $tables['advocacy_outcomes_' . $include_cases] = $this->getAdvocacyOutcomesTable();
                $tables['cancer_types_' . $include_cases] = $this->getCancerTypesTable();

                $tables['gender_' . $include_cases] = $this->getGenderTable();
                $tables['ethnicity_' . $include_cases] = $this->getEthnicityTable();
                $tables['sexuality_' . $include_cases] = $this->getSexualityTable();
                $tables['age_' . $include_cases] = $this->getAgeTable();
            }
            $this->include_cases = null;

            $this->tables = $tables;
        }
        return $this->tables;
    }

    // Bases

    public function getBaseQueryBuilder() {

        $query = parent::getBaseQueryBuilder();

        $query = $this->applySimpleComplexConstraints($query);

        return $query;

    }

    public function getBaseQueryBuilderForActivePeriodsWhere($where) {

        $query = $this->getTable();

        $query = $this->applyOrganisationConstraints($query);

        $query = $this->applyActivePeriodConstraints($query, $where);

        $query = $this->applySimpleComplexConstraints($query);

        return $query;
        
    }

    private function applySimpleComplexConstraints($query) {
        if($this->include_cases == 'simple' || $this->include_cases == 'complex') {
            $number_of_advocacy_issues = DB::table('advocacy_issues')
                ->selectRaw('count(*)')
                ->join('advocacy_issue_service_user_case', 'advocacy_issues.id', '=', 'advocacy_issue_service_user_case.advocacy_issue_id')
                ->whereRaw('`advocacy_issue_service_user_case`.`service_user_case_id` = `service_user_cases`.`id`');
            if($this->include_cases == 'simple') {
                $operator = '< 3';
            }
            if($this->include_cases == 'complex') {
                $operator = '>= 3';
            }
            $query = $query->whereRaw('(' . $number_of_advocacy_issues->toSql() . ') ' . $operator)->mergeBindings($number_of_advocacy_issues);
        }
        return $query;
    }

    protected function joinToPeopleTable($query) {
        return $query
            ->join('service_users', 'service_user_cases.service_user_id', '=', 'service_users.id')
            ->join('people', 'service_users.person_id', '=', 'people.id');
    }

    // Tables

    public function getTotals() {

        $total_opened = $this->getBaseQueryBuilder()
            ->selectRaw("
                count(service_user_cases.id) as cases,
                count(distinct service_users.id) as people
           ")
            ->first();

        $total_closed = $this->getBaseQueryBuilderForActivePeriodsWhere(function($query) {
            $query
                ->where('active_periods.end','>=',$this->from->format('Y-m-d'))
                ->where('active_periods.end','<=',$this->to->format('Y-m-d'));
            return $query;
        })->count();

        // Closure to find all active periods that are open at the end of the period (i.e. active period overlaps the end of the period)
        // Used in $open_right_now and $closed_right_now

        $open_right_now_closure = function($query) {
            $query
                ->whereNotNull('active_periods.start')
                ->where(function($query) {
                    $query->where('active_periods.start','<=',$this->to->format('Y-m-d'))
                        ->where('active_periods.end','>',$this->to->format('Y-m-d'))
                        ->orWhereNull('active_periods.end');
                });
            return $query;
        };

        // 1. Find all open right now by finding all with active periods matching $open_right_now_closure
        $open_right_now = $this->getBaseQueryBuilderForActivePeriodsWhere($open_right_now_closure);

        // 2. Find all closed right now by finding all that DON'T have any active periods matching $open_right_now
        // (i.e. no active periods that overlap the end of the project)
        $closed_right_now = $this->getTable();
        $closed_right_now = $this->applyOrganisationConstraints($closed_right_now);
        $closed_right_now = $this->applyNegativeActivePeriodConstraints($closed_right_now, $open_right_now_closure);
        $closed_right_now = $this->applySimpleComplexConstraints($closed_right_now);
        // This will include all closed and never opened, now need to do the below subquery to ensure only returning ones that
        // have been opened and closed (i.e. have active periods)
        $active_periods = $this->getActivePeriodsSubQuery();
        $closed_right_now = $closed_right_now->whereRaw('(' . $active_periods->toSql() . ') >= 1')->mergeBindings($active_periods);

        return new Table(
            'Totals (' . $this->include_cases . ' cases)',
            [],
            [
                [
                    'Total cases opened during this period',
                    $total_opened->cases
                ],
                [
                    'Total corresponding people (will be less than or equal to the above, since people can have multiple cases)',
                    $total_opened->people
                ],
                [
                    'Total cases closed during this period',
                    $total_closed
                ],
                [
                    'Total open cases in system',
                    // Includes the ones opened in this period
                    $open_right_now->count()
                ],
                [
                    'Total closed cases in system',
                    // Includes the ones closed in this period
                    $closed_right_now->count()
                ]
            ],
            null,
            function($row_index) {
                if($row_index == 3 || $row_index == 4) {
                    return route('sculpt.index',[
                        'service-user-cases',
                        'organisation_id' => $this->organisation_id,
                        'show_open' => $row_index == 3,
                        'show_closed' => $row_index == 4,
                        'show_never_opened' => 0
                    ]);
                }
            }
        );

    }

    public function getCaseTypesTable() {

        $_case_types = $this->getBaseQueryBuilder()
            ->selectRaw("service_user_cases.case_type, COUNT(*) as count")
            ->groupBy('service_user_cases.case_type');

        $case_types = [];

        foreach($_case_types->get() as $case_type) {
            if($case_type->case_type == null || $case_type->case_type == '') {
                $case_type->case_type = 'Undefined';
            }
            if(!isset($case_types[$case_type->case_type])) {
                $case_types[$case_type->case_type] = $case_type;
            } else {
                $case_types[$case_type->case_type]->count += $case_type->count;
            }
        }

        return new Table(
            'Case Type (' . $this->include_cases . ' cases)',
            [
                '',
                'Quantity'
            ],
            $case_types
        );
        
    }

    public function getActivitiesTable() {

        // TODO: Add breakdown of support hour groupings
        // TODO: make sure it includes ones that have no tasks yet

        // First get the subquery that finds all the cases and their total duration and number of activities

        $cases_duration_and_num = $this->getBaseQueryBuilder()
            ->selectRaw('
                service_user_cases.id as service_user_case_id,
                IF(advocates.type_of_advocate = "Volunteer", service_user_cases.id, 0) AS is_volunteer_advocate,
                IF(tasks.id IS NULL,0,(sum(tasks.duration)/60)) as total_hours,
                count(tasks.id) as total_num,
                IF(advocates.type_of_advocate = "Volunteer", count(tasks.id), 0) AS total_num_volunteer,
                IF(advocates.type_of_advocate = "Volunteer", IF(tasks.id IS NULL, 0, (sum(tasks.duration)/60)), 0) AS total_hours_volunteer
            ')
            ->leftJoin('tasks', 'tasks.service_user_case_id', '=', 'service_user_cases.id')
            ->leftJoin('employees', 'employees.user_id', '=', 'tasks.user_id')
            ->leftJoin('advocates', 'advocates.employee_id', '=', 'employees.id')
            ->where('tasks.date','>=',$this->from)
            ->where('tasks.date', '<=', $this->to)
            ->groupBy('tasks.id');

        // Next construct select statement to perform on this subquery in order to form the groups

        $time_boundaries = config('sam.reporting-support-time-boundaries');
        $time_boundaries_select = [];
        foreach($time_boundaries as $i => $cutoff) {
            if($i == 0) {
                $time_boundaries_select[] = "SUM(IF(cases_duration_and_num.total_hours < $cutoff, 1, 0)) as 'Fewer than $cutoff hours support'";
            } elseif($i == count($time_boundaries) - 1) {
                $time_boundaries_select[] = "SUM(IF(cases_duration_and_num.total_hours >= {$time_boundaries[$i-1]}, IF(cases_duration_and_num.total_hours < $cutoff, 1, 0), 0)) as 'From {$time_boundaries[$i-1]} up to $cutoff hours support'";
                $time_boundaries_select[] = "SUM(IF(cases_duration_and_num.total_hours >= $cutoff, 1, 0)) as '$cutoff hours support and over'";
            } else {
                $time_boundaries_select[] = "SUM(IF(cases_duration_and_num.total_hours >= {$time_boundaries[$i-1]}, IF(cases_duration_and_num.total_hours < $cutoff, 1, 0), 0)) as 'From {$time_boundaries[$i-1]} up to $cutoff hours support'";
            }
        }
        // Not needed in report but can uncomment to debug
//        $time_boundaries_select[] = "COUNT(cases_duration_and_num.service_user_case_id) as 'Total cases'";
        $time_boundaries_select[] = "(SUM(cases_duration_and_num.total_hours)*60) as 'Total support hours'";
        $time_boundaries_select[] = "(SUM(cases_duration_and_num.total_hours_volunteer)*60) as 'Total hours support received from volunteers'";
        $time_boundaries_select[] = "SUM(cases_duration_and_num.total_num) as 'Total number of activities'";
        $time_boundaries_select[] = "((SUM(cases_duration_and_num.total_hours)/COUNT(DISTINCT cases_duration_and_num.service_user_case_id))*60) as 'Average total support time per case'";
        $time_boundaries_select[] = "((SUM(cases_duration_and_num.total_hours_volunteer)/COUNT(DISTINCT cases_duration_and_num.is_volunteer_advocate))*60) as 'Average hours support received from volunteers'";
        $time_boundaries_select[] = "round(SUM(cases_duration_and_num.total_num)/COUNT(DISTINCT cases_duration_and_num.service_user_case_id)) as 'Average number of activities per case'";

        // Finally construct subquery and perform select statement to get stats

        $activities = DB::table(DB::raw('(' . $cases_duration_and_num->toSql() . ') as cases_duration_and_num'))
            ->selectRaw(implode(", ",$time_boundaries_select))
            ->mergeBindings($cases_duration_and_num)
            ->first();

        $rows = [];

        foreach(get_object_vars($activities) as $key => $value) {
            $rows[] = [$key,$value];
        }

        return new Table(
            'Activities (' . $this->include_cases . ' cases)',
            [],
            $rows,
            [
                (count($time_boundaries_select) - 3) => [
                    1 => $this->formatDuration()
                ],
                (count($time_boundaries_select) - 2) => [
                    1 => $this->formatDuration()
                ],
                (count($time_boundaries_select) - 6) => [
                    1 => $this->formatDuration()
                ],
                (count($time_boundaries_select) - 5) => [
                    1 => $this->formatDuration()
                ],
            ]
        );
    }

    public function getReferralTable() {

        list($counts, $ids) = $this->getTotalForRelationOnServiceUserCase('referral_source', 'referral_sources', 'referral_source_id', 'referral_source_other');

        return new Table(
            'Source (' . $this->include_cases . ' cases)',
            [
                '',
                'Quantity'
            ],
            $counts,
            null,
            function($row_index) use($ids) {
                return route('sculpt.index',[
                    'service-user-cases',
                    'organisation_id' => $this->organisation_id,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'referral_source' => $ids[$row_index]
                ]);
            }
        );

    }

    public function getHowTheyHeardTable() {
        return new Table(
            'How they heard (' . $this->include_cases . ' cases)',
            [
                '',
                'Quantity'
            ],
            $this->getTotalForEnumOnServiceUserCase('how_they_heard', 'how_they_heard_other'),
            null,
            function($row_index, $row) {
                return route('sculpt.index',[
                    'service-user-cases',
                    'organisation_id' => $this->organisation_id,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'how_they_heard' => $row->_how_they_heard !== 'Undefined' ? $row->_how_they_heard : 'Undefined',
                ]);
            }
        );
    }

    public function getReferralMethodTable() {

        list($counts, $ids) = $this->getTotalForRelationOnServiceUserCase('referral_method', 'referral_methods', 'referral_method_id', 'referral_method_other');

        return new Table(
            'Method (' . $this->include_cases . ' cases)',
            [
                '',
                'Quantity'
            ],
            $counts,
            null,
            function($row_index) use($ids) {
                return route('sculpt.index',[
                    'service-user-cases',
                    'organisation_id' => $this->organisation_id,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'referral_method' => $ids[$row_index]
                ]);
            }
        );

    }

    public function getReferralBreakdownTable() {

        $cases = $this->getBaseQueryBuilder()->selectRaw("
            service_user_cases.id as id,
            IFNULL(referral_sources.name, IF(service_user_cases.referral_source_other IS NULL,'Undefined',CONCAT('Other: ',service_user_cases.referral_source_other))) as referral_source,
            IFNULL(genders.name, 'Undefined') as gender,
            IF(people.age = 0, 'Undefined', IFNULL(people.age, 'Undefined')) as age
        ")
            ->leftJoin('genders','genders.id','=','people.gender_id')
            ->leftJoin('referral_sources','service_user_cases.referral_source_id','=','referral_sources.id')
            ->get();

        return new Table(
            'Referral breakdown (' . $this->include_cases . ' cases)',
            [
                'Case ID',
                'Referral source',
                'Gender',
                'Age'
            ],
            $cases,
            [
                '*' => [
                    0 => function($input) {
                        return '#' . $input; // add # in front of IDs
                    }
                ]
            ],
            function($row_index, $row) {
                return route('sculpt.show', ['service-user-cases', $row->id]);
            }
        );

    }

    public function getJourneyStageTable() {

        list($counts, $ids) = $this->getTotalForRelationOnServiceUserCase('journey_stage', 'journey_stages', 'journey_stage_id', 'journey_stage_other');

        return new Table(
            'Journey Stage (' . $this->include_cases . ' cases)',
            [
                '',
                'Quantity'
            ],
            $counts,
            null,
            function($row_index) use($ids) {
                return route('sculpt.index',[
                    'service-user-cases',
                    'organisation_id' => $this->organisation_id,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'journey_stage' => $ids[$row_index]
                ]);
            }
        );

    }

    public function getAdvocacyIssuesTable() {

        list($counts, $ids) = $this->getTotalForToManyRelationOnServiceUserCase('advocacy_issue', 'advocacy_issues', 'advocacy_issue_service_user_case', 'advocacy_issue_id', 'advocacy_issues_other');

        return new Table(
            'Advocacy Issues (' . $this->include_cases . ' cases)',
            [
                '',
                'Quantity'
            ],
            $counts,
            null,
            function($row_index) use($ids) {
                return route('sculpt.index',[
                    'service-user-cases',
                    'organisation_id' => $this->organisation_id,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'advocacy_issue' => $ids[$row_index]
                ]);
            }
        );

    }

    public function getAdvocacyOutcomesTable() {

        list($counts, $ids) = $this->getTotalForToManyRelationOnServiceUserCase('advocacy_outcome', 'advocacy_outcomes', 'advocacy_outcome_service_user_case', 'advocacy_outcome_id', null, function($query) {
            $query->where(function($query) {
                $query->where('advocacy_outcome_service_user_case.level','>',1)
                    ->orWhere(function($query) {
                        $query->whereNull('advocacy_outcome_service_user_case.advocacy_outcome_id');
                        // Only include cases that are closed (i.e. have no active periods where end date is not set)
                        $this->applyActivePeriodConstraints($query, function($query) {
                            return $query->whereNull('active_periods.end');
                        }, "< 1");
                    });
            });
            return $query;
        });

        return new Table(
            'Advocacy Outcomes (' . $this->include_cases . ' cases)',
            [
                '',
                'Quantity'
            ],
            $counts,
            null,
            function($row_index) use($ids) {
                return route('sculpt.index',[
                    'service-user-cases',
                    'organisation_id' => $this->organisation_id,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'advocacy_outcome' => $ids[$row_index]
                ]);
            }
        );

    }

    public function getCancerTypesTable() {

        list($counts, $ids) = $this->getTotalForToManyRelationOnServiceUserCase('cancer_type', 'cancer_types', 'cancer_type_service_user_case', 'cancer_type_id', 'cancer_types_other');

        return new Table(
            'Cancer Types (' . $this->include_cases . ' cases)',
            [
                '',
                'Quantity'
            ],
            $counts,
            null,
            function($row_index) use($ids) {
                return route('sculpt.index',[
                    'service-user-cases',
                    'organisation_id' => $this->organisation_id,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'cancer_type' => $ids[$row_index]
                ]);
            }
        );

    }

    public function getGenderTable() {

        $table = parent::getGenderTable();

        $ids = $this->ids;

        $table->setTitle('Gender (' . $this->include_cases . ' cases)');

        $table->setLinkMap(
            function($row_index) use($ids) {
                return route('sculpt.index',[
                    'service-user-cases',
                    'organisation_id' => $this->organisation_id,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'gender' => $ids[$row_index]
                ]);
            }
        );

        return $table;

    }

    public function getEthnicityTable() {

        $table = parent::getEthnicityTable();

        $table->setTitle('Ethnicity (' . $this->include_cases . ' cases)');

        $table->setLinkMap(
            function($row_index, $row) {
                return route('sculpt.index',[
                    'service-user-cases',
                    'organisation_id' => $this->organisation_id,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'ethnicity' => $row->_ethnicity !== 'Undefined' ? $row->_ethnicity : 'Undefined',
                ]);
            }
        );

        return $table;

    }

    public function getSexualityTable() {

        $table = parent::getSexualityTable();

        $ids = $this->ids;

        $table->setTitle('Sexuality (' . $this->include_cases . ' cases)');

        $table->setLinkMap(
            function($row_index) use($ids) {
                return route('sculpt.index',[
                    'service-user-cases',
                    'organisation_id' => $this->organisation_id,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'sexuality' => $ids[$row_index]
                ]);
            }
        );

        return $table;

    }

    public function getAgeTable() {

        $table = parent::getAgeTable();

        $table->setTitle('Age (' . $this->include_cases . ' cases)');

        $table->setLinkMap(
            function($row_index, $row) {
                return route('sculpt.index',[
                    'service-user-cases',
                    'organisation_id' => $this->organisation_id,
                    'date_from' => $this->from->format('Y-m'),
                    'date_to' => $this->to->format('Y-m'),
                    'age' => $row['group'] !== 'Undefined' ? $row['group'] : 'Undefined',
                ]);
            }
        );

        return $table;
    }

    // Helper functions

    private function getTotalForRelationOnServiceUserCase($relation_name, $relation_table, $foreign_key, $other_field_name) {

        $counts = $this->getBaseQueryBuilder()
            ->selectRaw("
                IFNULL($relation_table.id,
                    (CASE WHEN service_user_cases.$other_field_name IS NOT NULL THEN 'other' ELSE 'Undefined' END)
                ) as id,
                IFNULL($relation_table.name,
                    (CASE WHEN service_user_cases.$other_field_name IS NOT NULL THEN 'Other' ELSE 'Undefined' END)
                ) as $relation_name,
                count(*) as quantity
            ")
            ->leftJoin($relation_table, "service_user_cases.$foreign_key", '=', "$relation_table.id")
            ->groupBy($relation_name)
            ->get();

        $ids = array_pluck($counts,'id');
        foreach($counts as &$value) {
            unset($value->id);
        }

        return [$counts, $ids];

    }

    private function getTotalForToManyRelationOnServiceUserCase($relation_name, $relation_table, $join_table, $foreign_key, $other_field_name = null, $callback = null) {

        // Query 1: Actual values and Undefined (ignoring 'other')
        $query = $this->getBaseQueryBuilder()->selectRaw("
            IFNULL($relation_table.id,'Undefined') as $foreign_key,
            IFNULL($relation_table.name,'Undefined') as $relation_name,
            count(*) as quantity
        ")
            ->leftJoin($join_table, 'service_user_cases.id', '=', "$join_table.service_user_case_id")
            ->leftJoin($relation_table, "$join_table.$foreign_key", '=', "$relation_table.id");

        if($callback) {
            $query = $callback($query);
        }

        if($other_field_name) {
            $query = $query->where(function($query) use($other_field_name, $relation_table) {
                $query->whereNull($other_field_name)->orWhereNotNull("$relation_table.name");
            });
        }

        $query = $query->groupBy($relation_name);

        // Query 2: Count up 'other'. Need to do this separately as some of the above will have 'other', but we want to count the
        // ones with other in their own right
        if($other_field_name) {
            $other = $this->getBaseQueryBuilder()->selectRaw('
                "Other" as cancer_type_id,
                "Other" as cancer_type,
                count(*) as quantity
            ')
                ->whereNotNull('cancer_types_other');
            // Add to existing rows
            $query = $query->union($other);
        }

        $counts_raw = $query->get();
        $counts = $ids = [];
        foreach($counts_raw as $row) {
            $ids[] = $row->{$foreign_key};
            $counts[] = [
                $relation_name => $row->{$relation_name},
                'quantity' => $row->quantity
            ];
        }

        return [$counts, $ids];

    }

    private function getTotalForEnumOnServiceUserCase($field, $other_field_name) {

        $counts = $this->getBaseQueryBuilder()
            ->selectRaw("
                IFNULL($field,
                    (CASE WHEN $other_field_name IS NOT NULL THEN 'Other' ELSE 'Undefined' END)
                ) as _$field,
                count(*) as quantity
            ")
            ->groupBy("_$field")
            ->get();

        return $counts;

    }

}